/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
import * as dom from '../../../base/browser/dom';
import { Color } from '../../../base/common/color';
import { Emitter } from '../../../base/common/event';
import { TokenizationRegistry, TokenMetadata } from '../../common/modes';
import { TokenTheme, generateTokensCSSForColorMap } from '../../common/modes/supports/tokenization';
import { hc_black, vs, vs_dark } from '../common/themes';
import { Registry } from '../../../platform/registry/common/platform';
import { Extensions } from '../../../platform/theme/common/colorRegistry';
import { Extensions as ThemingExtensions } from '../../../platform/theme/common/themeService';
import { Disposable } from '../../../base/common/lifecycle';
import { ColorScheme } from '../../../platform/theme/common/theme';
import { getIconsStyleSheet } from '../../../platform/theme/browser/iconsStyleSheet';
var VS_THEME_NAME = 'vs';
var VS_DARK_THEME_NAME = 'vs-dark';
var HC_BLACK_THEME_NAME = 'hc-black';
var colorRegistry = Registry.as(Extensions.ColorContribution);
var themingRegistry = Registry.as(ThemingExtensions.ThemingContribution);
var StandaloneTheme = /** @class */ (function () {
    function StandaloneTheme(name, standaloneThemeData) {
        this.semanticHighlighting = false;
        this.themeData = standaloneThemeData;
        var base = standaloneThemeData.base;
        if (name.length > 0) {
            if (isBuiltinTheme(name)) {
                this.id = name;
            }
            else {
                this.id = base + ' ' + name;
            }
            this.themeName = name;
        }
        else {
            this.id = base;
            this.themeName = base;
        }
        this.colors = null;
        this.defaultColors = Object.create(null);
        this._tokenTheme = null;
    }
    Object.defineProperty(StandaloneTheme.prototype, "label", {
        get: function () {
            return this.themeName;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(StandaloneTheme.prototype, "base", {
        get: function () {
            return this.themeData.base;
        },
        enumerable: false,
        configurable: true
    });
    StandaloneTheme.prototype.notifyBaseUpdated = function () {
        if (this.themeData.inherit) {
            this.colors = null;
            this._tokenTheme = null;
        }
    };
    StandaloneTheme.prototype.getColors = function () {
        if (!this.colors) {
            var colors = new Map();
            for (var id in this.themeData.colors) {
                colors.set(id, Color.fromHex(this.themeData.colors[id]));
            }
            if (this.themeData.inherit) {
                var baseData = getBuiltinRules(this.themeData.base);
                for (var id in baseData.colors) {
                    if (!colors.has(id)) {
                        colors.set(id, Color.fromHex(baseData.colors[id]));
                    }
                }
            }
            this.colors = colors;
        }
        return this.colors;
    };
    StandaloneTheme.prototype.getColor = function (colorId, useDefault) {
        var color = this.getColors().get(colorId);
        if (color) {
            return color;
        }
        if (useDefault !== false) {
            return this.getDefault(colorId);
        }
        return undefined;
    };
    StandaloneTheme.prototype.getDefault = function (colorId) {
        var color = this.defaultColors[colorId];
        if (color) {
            return color;
        }
        color = colorRegistry.resolveDefaultColor(colorId, this);
        this.defaultColors[colorId] = color;
        return color;
    };
    StandaloneTheme.prototype.defines = function (colorId) {
        return Object.prototype.hasOwnProperty.call(this.getColors(), colorId);
    };
    Object.defineProperty(StandaloneTheme.prototype, "type", {
        get: function () {
            switch (this.base) {
                case VS_THEME_NAME: return ColorScheme.LIGHT;
                case HC_BLACK_THEME_NAME: return ColorScheme.HIGH_CONTRAST;
                default: return ColorScheme.DARK;
            }
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(StandaloneTheme.prototype, "tokenTheme", {
        get: function () {
            if (!this._tokenTheme) {
                var rules = [];
                var encodedTokensColors = [];
                if (this.themeData.inherit) {
                    var baseData = getBuiltinRules(this.themeData.base);
                    rules = baseData.rules;
                    if (baseData.encodedTokensColors) {
                        encodedTokensColors = baseData.encodedTokensColors;
                    }
                }
                rules = rules.concat(this.themeData.rules);
                if (this.themeData.encodedTokensColors) {
                    encodedTokensColors = this.themeData.encodedTokensColors;
                }
                this._tokenTheme = TokenTheme.createFromRawTokenTheme(rules, encodedTokensColors);
            }
            return this._tokenTheme;
        },
        enumerable: false,
        configurable: true
    });
    StandaloneTheme.prototype.getTokenStyleMetadata = function (type, modifiers, modelLanguage) {
        // use theme rules match
        var style = this.tokenTheme._match([type].concat(modifiers).join('.'));
        var metadata = style.metadata;
        var foreground = TokenMetadata.getForeground(metadata);
        var fontStyle = TokenMetadata.getFontStyle(metadata);
        return {
            foreground: foreground,
            italic: Boolean(fontStyle & 1 /* Italic */),
            bold: Boolean(fontStyle & 2 /* Bold */),
            underline: Boolean(fontStyle & 4 /* Underline */)
        };
    };
    Object.defineProperty(StandaloneTheme.prototype, "tokenColorMap", {
        get: function () {
            return [];
        },
        enumerable: false,
        configurable: true
    });
    return StandaloneTheme;
}());
function isBuiltinTheme(themeName) {
    return (themeName === VS_THEME_NAME
        || themeName === VS_DARK_THEME_NAME
        || themeName === HC_BLACK_THEME_NAME);
}
function getBuiltinRules(builtinTheme) {
    switch (builtinTheme) {
        case VS_THEME_NAME:
            return vs;
        case VS_DARK_THEME_NAME:
            return vs_dark;
        case HC_BLACK_THEME_NAME:
            return hc_black;
    }
}
function newBuiltInTheme(builtinTheme) {
    var themeData = getBuiltinRules(builtinTheme);
    return new StandaloneTheme(builtinTheme, themeData);
}
var StandaloneThemeServiceImpl = /** @class */ (function (_super) {
    __extends(StandaloneThemeServiceImpl, _super);
    function StandaloneThemeServiceImpl() {
        var _this = _super.call(this) || this;
        _this._onColorThemeChange = _this._register(new Emitter());
        _this.onDidColorThemeChange = _this._onColorThemeChange.event;
        _this._onFileIconThemeChange = _this._register(new Emitter());
        _this.onDidFileIconThemeChange = _this._onFileIconThemeChange.event;
        _this._environment = Object.create(null);
        _this._autoDetectHighContrast = true;
        _this._knownThemes = new Map();
        _this._knownThemes.set(VS_THEME_NAME, newBuiltInTheme(VS_THEME_NAME));
        _this._knownThemes.set(VS_DARK_THEME_NAME, newBuiltInTheme(VS_DARK_THEME_NAME));
        _this._knownThemes.set(HC_BLACK_THEME_NAME, newBuiltInTheme(HC_BLACK_THEME_NAME));
        var iconsStyleSheet = getIconsStyleSheet();
        _this._codiconCSS = iconsStyleSheet.getCSS();
        _this._themeCSS = '';
        _this._allCSS = _this._codiconCSS + "\n" + _this._themeCSS;
        _this._globalStyleElement = null;
        _this._styleElements = [];
        _this._colorMapOverride = null;
        _this.setTheme(VS_THEME_NAME);
        iconsStyleSheet.onDidChange(function () {
            _this._codiconCSS = iconsStyleSheet.getCSS();
            _this._updateCSS();
        });
        window.matchMedia('(forced-colors: active)').addEventListener('change', function () {
            _this._updateActualTheme();
        });
        return _this;
    }
    StandaloneThemeServiceImpl.prototype.registerEditorContainer = function (domNode) {
        if (dom.isInShadowDOM(domNode)) {
            return this._registerShadowDomContainer(domNode);
        }
        return this._registerRegularEditorContainer();
    };
    StandaloneThemeServiceImpl.prototype._registerRegularEditorContainer = function () {
        if (!this._globalStyleElement) {
            this._globalStyleElement = dom.createStyleSheet();
            this._globalStyleElement.className = 'monaco-colors';
            this._globalStyleElement.textContent = this._allCSS;
            this._styleElements.push(this._globalStyleElement);
        }
        return Disposable.None;
    };
    StandaloneThemeServiceImpl.prototype._registerShadowDomContainer = function (domNode) {
        var _this = this;
        var styleElement = dom.createStyleSheet(domNode);
        styleElement.className = 'monaco-colors';
        styleElement.textContent = this._allCSS;
        this._styleElements.push(styleElement);
        return {
            dispose: function () {
                for (var i = 0; i < _this._styleElements.length; i++) {
                    if (_this._styleElements[i] === styleElement) {
                        _this._styleElements.splice(i, 1);
                        return;
                    }
                }
            }
        };
    };
    StandaloneThemeServiceImpl.prototype.defineTheme = function (themeName, themeData) {
        if (!/^[a-z0-9\-]+$/i.test(themeName)) {
            throw new Error('Illegal theme name!');
        }
        if (!isBuiltinTheme(themeData.base) && !isBuiltinTheme(themeName)) {
            throw new Error('Illegal theme base!');
        }
        // set or replace theme
        this._knownThemes.set(themeName, new StandaloneTheme(themeName, themeData));
        if (isBuiltinTheme(themeName)) {
            this._knownThemes.forEach(function (theme) {
                if (theme.base === themeName) {
                    theme.notifyBaseUpdated();
                }
            });
        }
        if (this._theme.themeName === themeName) {
            this.setTheme(themeName); // refresh theme
        }
    };
    StandaloneThemeServiceImpl.prototype.getColorTheme = function () {
        return this._theme;
    };
    StandaloneThemeServiceImpl.prototype.setColorMapOverride = function (colorMapOverride) {
        this._colorMapOverride = colorMapOverride;
        this._updateThemeOrColorMap();
    };
    StandaloneThemeServiceImpl.prototype.setTheme = function (themeName) {
        var theme;
        if (this._knownThemes.has(themeName)) {
            theme = this._knownThemes.get(themeName);
        }
        else {
            theme = this._knownThemes.get(VS_THEME_NAME);
        }
        this._desiredTheme = theme;
        this._updateActualTheme();
    };
    StandaloneThemeServiceImpl.prototype._updateActualTheme = function () {
        var theme = (this._autoDetectHighContrast && window.matchMedia("(forced-colors: active)").matches
            ? this._knownThemes.get(HC_BLACK_THEME_NAME)
            : this._desiredTheme);
        if (this._theme === theme) {
            // Nothing to do
            return;
        }
        this._theme = theme;
        this._updateThemeOrColorMap();
    };
    StandaloneThemeServiceImpl.prototype.setAutoDetectHighContrast = function (autoDetectHighContrast) {
        this._autoDetectHighContrast = autoDetectHighContrast;
        this._updateActualTheme();
    };
    StandaloneThemeServiceImpl.prototype._updateThemeOrColorMap = function () {
        var _this = this;
        var cssRules = [];
        var hasRule = {};
        var ruleCollector = {
            addRule: function (rule) {
                if (!hasRule[rule]) {
                    cssRules.push(rule);
                    hasRule[rule] = true;
                }
            }
        };
        themingRegistry.getThemingParticipants().forEach(function (p) { return p(_this._theme, ruleCollector, _this._environment); });
        var colorMap = this._colorMapOverride || this._theme.tokenTheme.getColorMap();
        ruleCollector.addRule(generateTokensCSSForColorMap(colorMap));
        this._themeCSS = cssRules.join('\n');
        this._updateCSS();
        TokenizationRegistry.setColorMap(colorMap);
        this._onColorThemeChange.fire(this._theme);
    };
    StandaloneThemeServiceImpl.prototype._updateCSS = function () {
        var _this = this;
        this._allCSS = this._codiconCSS + "\n" + this._themeCSS;
        this._styleElements.forEach(function (styleElement) { return styleElement.textContent = _this._allCSS; });
    };
    StandaloneThemeServiceImpl.prototype.getFileIconTheme = function () {
        return {
            hasFileIcons: false,
            hasFolderIcons: false,
            hidesExplorerArrows: false
        };
    };
    return StandaloneThemeServiceImpl;
}(Disposable));
export { StandaloneThemeServiceImpl };
