/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
import { binarySearch, coalesceInPlace, equals } from '../../../base/common/arrays';
import { CancellationTokenSource } from '../../../base/common/cancellation';
import { onUnexpectedExternalError } from '../../../base/common/errors';
import { LRUCache } from '../../../base/common/map';
import { commonPrefixLength } from '../../../base/common/strings';
import { Range } from '../../common/core/range';
import { DocumentSymbolProviderRegistry } from '../../common/modes';
import { Iterable } from '../../../base/common/iterator';
import { LanguageFeatureRequestDelays } from '../../common/modes/languageFeatureRegistry';
var TreeElement = /** @class */ (function () {
    function TreeElement() {
    }
    TreeElement.prototype.remove = function () {
        if (this.parent) {
            this.parent.children.delete(this.id);
        }
    };
    TreeElement.findId = function (candidate, container) {
        // complex id-computation which contains the origin/extension,
        // the parent path, and some dedupe logic when names collide
        var candidateId;
        if (typeof candidate === 'string') {
            candidateId = container.id + "/" + candidate;
        }
        else {
            candidateId = container.id + "/" + candidate.name;
            if (container.children.get(candidateId) !== undefined) {
                candidateId = container.id + "/" + candidate.name + "_" + candidate.range.startLineNumber + "_" + candidate.range.startColumn;
            }
        }
        var id = candidateId;
        for (var i = 0; container.children.get(id) !== undefined; i++) {
            id = candidateId + "_" + i;
        }
        return id;
    };
    TreeElement.getElementById = function (id, element) {
        var e_1, _a;
        if (!id) {
            return undefined;
        }
        var len = commonPrefixLength(id, element.id);
        if (len === id.length) {
            return element;
        }
        if (len < element.id.length) {
            return undefined;
        }
        try {
            for (var _b = __values(element.children), _c = _b.next(); !_c.done; _c = _b.next()) {
                var _d = __read(_c.value, 2), child = _d[1];
                var candidate = TreeElement.getElementById(id, child);
                if (candidate) {
                    return candidate;
                }
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_1) throw e_1.error; }
        }
        return undefined;
    };
    TreeElement.size = function (element) {
        var e_2, _a;
        var res = 1;
        try {
            for (var _b = __values(element.children), _c = _b.next(); !_c.done; _c = _b.next()) {
                var _d = __read(_c.value, 2), child = _d[1];
                res += TreeElement.size(child);
            }
        }
        catch (e_2_1) { e_2 = { error: e_2_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_2) throw e_2.error; }
        }
        return res;
    };
    TreeElement.empty = function (element) {
        return element.children.size === 0;
    };
    return TreeElement;
}());
export { TreeElement };
var OutlineElement = /** @class */ (function (_super) {
    __extends(OutlineElement, _super);
    function OutlineElement(id, parent, symbol) {
        var _this = _super.call(this) || this;
        _this.id = id;
        _this.parent = parent;
        _this.symbol = symbol;
        _this.children = new Map();
        return _this;
    }
    OutlineElement.prototype.adopt = function (parent) {
        var e_3, _a;
        var res = new OutlineElement(this.id, parent, this.symbol);
        try {
            for (var _b = __values(this.children), _c = _b.next(); !_c.done; _c = _b.next()) {
                var _d = __read(_c.value, 2), key = _d[0], value = _d[1];
                res.children.set(key, value.adopt(res));
            }
        }
        catch (e_3_1) { e_3 = { error: e_3_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_3) throw e_3.error; }
        }
        return res;
    };
    return OutlineElement;
}(TreeElement));
export { OutlineElement };
var OutlineGroup = /** @class */ (function (_super) {
    __extends(OutlineGroup, _super);
    function OutlineGroup(id, parent, label, order) {
        var _this = _super.call(this) || this;
        _this.id = id;
        _this.parent = parent;
        _this.label = label;
        _this.order = order;
        _this.children = new Map();
        return _this;
    }
    OutlineGroup.prototype.adopt = function (parent) {
        var e_4, _a;
        var res = new OutlineGroup(this.id, parent, this.label, this.order);
        try {
            for (var _b = __values(this.children), _c = _b.next(); !_c.done; _c = _b.next()) {
                var _d = __read(_c.value, 2), key = _d[0], value = _d[1];
                res.children.set(key, value.adopt(res));
            }
        }
        catch (e_4_1) { e_4 = { error: e_4_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_4) throw e_4.error; }
        }
        return res;
    };
    OutlineGroup.prototype.getItemEnclosingPosition = function (position) {
        return position ? this._getItemEnclosingPosition(position, this.children) : undefined;
    };
    OutlineGroup.prototype._getItemEnclosingPosition = function (position, children) {
        var e_5, _a;
        try {
            for (var children_1 = __values(children), children_1_1 = children_1.next(); !children_1_1.done; children_1_1 = children_1.next()) {
                var _b = __read(children_1_1.value, 2), item = _b[1];
                if (!item.symbol.range || !Range.containsPosition(item.symbol.range, position)) {
                    continue;
                }
                return this._getItemEnclosingPosition(position, item.children) || item;
            }
        }
        catch (e_5_1) { e_5 = { error: e_5_1 }; }
        finally {
            try {
                if (children_1_1 && !children_1_1.done && (_a = children_1.return)) _a.call(children_1);
            }
            finally { if (e_5) throw e_5.error; }
        }
        return undefined;
    };
    OutlineGroup.prototype.updateMarker = function (marker) {
        var e_6, _a;
        try {
            for (var _b = __values(this.children), _c = _b.next(); !_c.done; _c = _b.next()) {
                var _d = __read(_c.value, 2), child = _d[1];
                this._updateMarker(marker, child);
            }
        }
        catch (e_6_1) { e_6 = { error: e_6_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_6) throw e_6.error; }
        }
    };
    OutlineGroup.prototype._updateMarker = function (markers, item) {
        var e_7, _a;
        item.marker = undefined;
        // find the proper start index to check for item/marker overlap.
        var idx = binarySearch(markers, item.symbol.range, Range.compareRangesUsingStarts);
        var start;
        if (idx < 0) {
            start = ~idx;
            if (start > 0 && Range.areIntersecting(markers[start - 1], item.symbol.range)) {
                start -= 1;
            }
        }
        else {
            start = idx;
        }
        var myMarkers = [];
        var myTopSev;
        for (; start < markers.length && Range.areIntersecting(item.symbol.range, markers[start]); start++) {
            // remove markers intersecting with this outline element
            // and store them in a 'private' array.
            var marker = markers[start];
            myMarkers.push(marker);
            markers[start] = undefined;
            if (!myTopSev || marker.severity > myTopSev) {
                myTopSev = marker.severity;
            }
        }
        try {
            // Recurse into children and let them match markers that have matched
            // this outline element. This might remove markers from this element and
            // therefore we remember that we have had markers. That allows us to render
            // the dot, saying 'this element has children with markers'
            for (var _b = __values(item.children), _c = _b.next(); !_c.done; _c = _b.next()) {
                var _d = __read(_c.value, 2), child = _d[1];
                this._updateMarker(myMarkers, child);
            }
        }
        catch (e_7_1) { e_7 = { error: e_7_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_7) throw e_7.error; }
        }
        if (myTopSev) {
            item.marker = {
                count: myMarkers.length,
                topSev: myTopSev
            };
        }
        coalesceInPlace(markers);
    };
    return OutlineGroup;
}(TreeElement));
export { OutlineGroup };
var OutlineModel = /** @class */ (function (_super) {
    __extends(OutlineModel, _super);
    function OutlineModel(uri) {
        var _this = _super.call(this) || this;
        _this.uri = uri;
        _this.id = 'root';
        _this.parent = undefined;
        _this._groups = new Map();
        _this.children = new Map();
        _this.id = 'root';
        _this.parent = undefined;
        return _this;
    }
    OutlineModel.create = function (textModel, token) {
        var _this = this;
        var key = this._keys.for(textModel, true);
        var data = OutlineModel._requests.get(key);
        if (!data) {
            var source = new CancellationTokenSource();
            data = {
                promiseCnt: 0,
                source: source,
                promise: OutlineModel._create(textModel, source.token),
                model: undefined,
            };
            OutlineModel._requests.set(key, data);
            // keep moving average of request durations
            var now_1 = Date.now();
            data.promise.then(function () {
                _this._requestDurations.update(textModel, Date.now() - now_1);
            });
        }
        if (data.model) {
            // resolved -> return data
            return Promise.resolve(data.model);
        }
        // increase usage counter
        data.promiseCnt += 1;
        token.onCancellationRequested(function () {
            // last -> cancel provider request, remove cached promise
            if (--data.promiseCnt === 0) {
                data.source.cancel();
                OutlineModel._requests.delete(key);
            }
        });
        return new Promise(function (resolve, reject) {
            data.promise.then(function (model) {
                data.model = model;
                resolve(model);
            }, function (err) {
                OutlineModel._requests.delete(key);
                reject(err);
            });
        });
    };
    OutlineModel.getRequestDelay = function (textModel) {
        return textModel ? this._requestDurations.get(textModel) : this._requestDurations.min;
    };
    OutlineModel._create = function (textModel, token) {
        var cts = new CancellationTokenSource(token);
        var result = new OutlineModel(textModel.uri);
        var provider = DocumentSymbolProviderRegistry.ordered(textModel);
        var promises = provider.map(function (provider, index) {
            var _a;
            var id = TreeElement.findId("provider_" + index, result);
            var group = new OutlineGroup(id, result, (_a = provider.displayName) !== null && _a !== void 0 ? _a : 'Unknown Outline Provider', index);
            return Promise.resolve(provider.provideDocumentSymbols(textModel, cts.token)).then(function (result) {
                var e_8, _a;
                try {
                    for (var _b = __values(result || []), _c = _b.next(); !_c.done; _c = _b.next()) {
                        var info = _c.value;
                        OutlineModel._makeOutlineElement(info, group);
                    }
                }
                catch (e_8_1) { e_8 = { error: e_8_1 }; }
                finally {
                    try {
                        if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                    }
                    finally { if (e_8) throw e_8.error; }
                }
                return group;
            }, function (err) {
                onUnexpectedExternalError(err);
                return group;
            }).then(function (group) {
                if (!TreeElement.empty(group)) {
                    result._groups.set(id, group);
                }
                else {
                    group.remove();
                }
            });
        });
        var listener = DocumentSymbolProviderRegistry.onDidChange(function () {
            var newProvider = DocumentSymbolProviderRegistry.ordered(textModel);
            if (!equals(newProvider, provider)) {
                cts.cancel();
            }
        });
        return Promise.all(promises).then(function () {
            if (cts.token.isCancellationRequested && !token.isCancellationRequested) {
                return OutlineModel._create(textModel, token);
            }
            else {
                return result._compact();
            }
        }).finally(function () {
            listener.dispose();
        });
    };
    OutlineModel._makeOutlineElement = function (info, container) {
        var e_9, _a;
        var id = TreeElement.findId(info, container);
        var res = new OutlineElement(id, container, info);
        if (info.children) {
            try {
                for (var _b = __values(info.children), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var childInfo = _c.value;
                    OutlineModel._makeOutlineElement(childInfo, res);
                }
            }
            catch (e_9_1) { e_9 = { error: e_9_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_9) throw e_9.error; }
            }
        }
        container.children.set(res.id, res);
    };
    OutlineModel.get = function (element) {
        while (element) {
            if (element instanceof OutlineModel) {
                return element;
            }
            element = element.parent;
        }
        return undefined;
    };
    OutlineModel.prototype.adopt = function () {
        var e_10, _a;
        var res = new OutlineModel(this.uri);
        try {
            for (var _b = __values(this._groups), _c = _b.next(); !_c.done; _c = _b.next()) {
                var _d = __read(_c.value, 2), key = _d[0], value = _d[1];
                res._groups.set(key, value.adopt(res));
            }
        }
        catch (e_10_1) { e_10 = { error: e_10_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_10) throw e_10.error; }
        }
        return res._compact();
    };
    OutlineModel.prototype._compact = function () {
        var e_11, _a, e_12, _b;
        var count = 0;
        try {
            for (var _c = __values(this._groups), _d = _c.next(); !_d.done; _d = _c.next()) {
                var _e = __read(_d.value, 2), key = _e[0], group = _e[1];
                if (group.children.size === 0) { // empty
                    this._groups.delete(key);
                }
                else {
                    count += 1;
                }
            }
        }
        catch (e_11_1) { e_11 = { error: e_11_1 }; }
        finally {
            try {
                if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
            }
            finally { if (e_11) throw e_11.error; }
        }
        if (count !== 1) {
            //
            this.children = this._groups;
        }
        else {
            // adopt all elements of the first group
            var group = Iterable.first(this._groups.values());
            try {
                for (var _f = __values(group.children), _g = _f.next(); !_g.done; _g = _f.next()) {
                    var _h = __read(_g.value, 2), child = _h[1];
                    child.parent = this;
                    this.children.set(child.id, child);
                }
            }
            catch (e_12_1) { e_12 = { error: e_12_1 }; }
            finally {
                try {
                    if (_g && !_g.done && (_b = _f.return)) _b.call(_f);
                }
                finally { if (e_12) throw e_12.error; }
            }
        }
        return this;
    };
    OutlineModel.prototype.merge = function (other) {
        if (this.uri.toString() !== other.uri.toString()) {
            return false;
        }
        if (this._groups.size !== other._groups.size) {
            return false;
        }
        this._groups = other._groups;
        this.children = other.children;
        return true;
    };
    OutlineModel.prototype.getItemEnclosingPosition = function (position, context) {
        var e_13, _a;
        var preferredGroup;
        if (context) {
            var candidate = context.parent;
            while (candidate && !preferredGroup) {
                if (candidate instanceof OutlineGroup) {
                    preferredGroup = candidate;
                }
                candidate = candidate.parent;
            }
        }
        var result = undefined;
        try {
            for (var _b = __values(this._groups), _c = _b.next(); !_c.done; _c = _b.next()) {
                var _d = __read(_c.value, 2), group = _d[1];
                result = group.getItemEnclosingPosition(position);
                if (result && (!preferredGroup || preferredGroup === group)) {
                    break;
                }
            }
        }
        catch (e_13_1) { e_13 = { error: e_13_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_13) throw e_13.error; }
        }
        return result;
    };
    OutlineModel.prototype.getItemById = function (id) {
        return TreeElement.getElementById(id, this);
    };
    OutlineModel.prototype.updateMarker = function (marker) {
        var e_14, _a;
        // sort markers by start range so that we can use
        // outline element starts for quicker look up
        marker.sort(Range.compareRangesUsingStarts);
        try {
            for (var _b = __values(this._groups), _c = _b.next(); !_c.done; _c = _b.next()) {
                var _d = __read(_c.value, 2), group = _d[1];
                group.updateMarker(marker.slice(0));
            }
        }
        catch (e_14_1) { e_14 = { error: e_14_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_14) throw e_14.error; }
        }
    };
    OutlineModel.prototype.getTopLevelSymbols = function () {
        var e_15, _a;
        var roots = [];
        try {
            for (var _b = __values(this.children.values()), _c = _b.next(); !_c.done; _c = _b.next()) {
                var child = _c.value;
                if (child instanceof OutlineElement) {
                    roots.push(child.symbol);
                }
                else {
                    roots.push.apply(roots, __spreadArray([], __read(Iterable.map(child.children.values(), function (child) { return child.symbol; }))));
                }
            }
        }
        catch (e_15_1) { e_15 = { error: e_15_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_15) throw e_15.error; }
        }
        return roots.sort(function (a, b) { return Range.compareRangesUsingStarts(a.range, b.range); });
    };
    OutlineModel.prototype.asListOfDocumentSymbols = function () {
        var roots = this.getTopLevelSymbols();
        var bucket = [];
        OutlineModel._flattenDocumentSymbols(bucket, roots, '');
        return bucket.sort(function (a, b) { return Range.compareRangesUsingStarts(a.range, b.range); });
    };
    OutlineModel._flattenDocumentSymbols = function (bucket, entries, overrideContainerLabel) {
        var e_16, _a;
        try {
            for (var entries_1 = __values(entries), entries_1_1 = entries_1.next(); !entries_1_1.done; entries_1_1 = entries_1.next()) {
                var entry = entries_1_1.value;
                bucket.push({
                    kind: entry.kind,
                    tags: entry.tags,
                    name: entry.name,
                    detail: entry.detail,
                    containerName: entry.containerName || overrideContainerLabel,
                    range: entry.range,
                    selectionRange: entry.selectionRange,
                    children: undefined, // we flatten it...
                });
                // Recurse over children
                if (entry.children) {
                    OutlineModel._flattenDocumentSymbols(bucket, entry.children, entry.name);
                }
            }
        }
        catch (e_16_1) { e_16 = { error: e_16_1 }; }
        finally {
            try {
                if (entries_1_1 && !entries_1_1.done && (_a = entries_1.return)) _a.call(entries_1);
            }
            finally { if (e_16) throw e_16.error; }
        }
    };
    OutlineModel._requestDurations = new LanguageFeatureRequestDelays(DocumentSymbolProviderRegistry, 350);
    OutlineModel._requests = new LRUCache(9, 0.75);
    OutlineModel._keys = new /** @class */ (function () {
        function class_1() {
            this._counter = 1;
            this._data = new WeakMap();
        }
        class_1.prototype.for = function (textModel, version) {
            return textModel.id + "/" + (version ? textModel.getVersionId() : '') + "/" + this._hash(DocumentSymbolProviderRegistry.all(textModel));
        };
        class_1.prototype._hash = function (providers) {
            var e_17, _a;
            var result = '';
            try {
                for (var providers_1 = __values(providers), providers_1_1 = providers_1.next(); !providers_1_1.done; providers_1_1 = providers_1.next()) {
                    var provider = providers_1_1.value;
                    var n = this._data.get(provider);
                    if (typeof n === 'undefined') {
                        n = this._counter++;
                        this._data.set(provider, n);
                    }
                    result += n;
                }
            }
            catch (e_17_1) { e_17 = { error: e_17_1 }; }
            finally {
                try {
                    if (providers_1_1 && !providers_1_1.done && (_a = providers_1.return)) _a.call(providers_1);
                }
                finally { if (e_17) throw e_17.error; }
            }
            return result;
        };
        return class_1;
    }());
    return OutlineModel;
}(TreeElement));
export { OutlineModel };
