/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import * as nls from '../../../nls';
import { onUnexpectedError } from '../../../base/common/errors';
import { Selection } from '../core/selection';
import { URI } from '../../../base/common/uri';
import { TextChange, compressConsecutiveTextChanges } from './textChange';
import * as buffer from '../../../base/common/buffer';
import { basename } from '../../../base/common/resources';
function uriGetComparisonKey(resource) {
    return resource.toString();
}
var SingleModelEditStackData = /** @class */ (function () {
    function SingleModelEditStackData(beforeVersionId, afterVersionId, beforeEOL, afterEOL, beforeCursorState, afterCursorState, changes) {
        this.beforeVersionId = beforeVersionId;
        this.afterVersionId = afterVersionId;
        this.beforeEOL = beforeEOL;
        this.afterEOL = afterEOL;
        this.beforeCursorState = beforeCursorState;
        this.afterCursorState = afterCursorState;
        this.changes = changes;
    }
    SingleModelEditStackData.create = function (model, beforeCursorState) {
        var alternativeVersionId = model.getAlternativeVersionId();
        var eol = getModelEOL(model);
        return new SingleModelEditStackData(alternativeVersionId, alternativeVersionId, eol, eol, beforeCursorState, beforeCursorState, []);
    };
    SingleModelEditStackData.prototype.append = function (model, textChanges, afterEOL, afterVersionId, afterCursorState) {
        if (textChanges.length > 0) {
            this.changes = compressConsecutiveTextChanges(this.changes, textChanges);
        }
        this.afterEOL = afterEOL;
        this.afterVersionId = afterVersionId;
        this.afterCursorState = afterCursorState;
    };
    SingleModelEditStackData._writeSelectionsSize = function (selections) {
        return 4 + 4 * 4 * (selections ? selections.length : 0);
    };
    SingleModelEditStackData._writeSelections = function (b, selections, offset) {
        var e_1, _a;
        buffer.writeUInt32BE(b, (selections ? selections.length : 0), offset);
        offset += 4;
        if (selections) {
            try {
                for (var selections_1 = __values(selections), selections_1_1 = selections_1.next(); !selections_1_1.done; selections_1_1 = selections_1.next()) {
                    var selection = selections_1_1.value;
                    buffer.writeUInt32BE(b, selection.selectionStartLineNumber, offset);
                    offset += 4;
                    buffer.writeUInt32BE(b, selection.selectionStartColumn, offset);
                    offset += 4;
                    buffer.writeUInt32BE(b, selection.positionLineNumber, offset);
                    offset += 4;
                    buffer.writeUInt32BE(b, selection.positionColumn, offset);
                    offset += 4;
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (selections_1_1 && !selections_1_1.done && (_a = selections_1.return)) _a.call(selections_1);
                }
                finally { if (e_1) throw e_1.error; }
            }
        }
        return offset;
    };
    SingleModelEditStackData._readSelections = function (b, offset, dest) {
        var count = buffer.readUInt32BE(b, offset);
        offset += 4;
        for (var i = 0; i < count; i++) {
            var selectionStartLineNumber = buffer.readUInt32BE(b, offset);
            offset += 4;
            var selectionStartColumn = buffer.readUInt32BE(b, offset);
            offset += 4;
            var positionLineNumber = buffer.readUInt32BE(b, offset);
            offset += 4;
            var positionColumn = buffer.readUInt32BE(b, offset);
            offset += 4;
            dest.push(new Selection(selectionStartLineNumber, selectionStartColumn, positionLineNumber, positionColumn));
        }
        return offset;
    };
    SingleModelEditStackData.prototype.serialize = function () {
        var e_2, _a, e_3, _b;
        var necessarySize = (+4 // beforeVersionId
            + 4 // afterVersionId
            + 1 // beforeEOL
            + 1 // afterEOL
            + SingleModelEditStackData._writeSelectionsSize(this.beforeCursorState)
            + SingleModelEditStackData._writeSelectionsSize(this.afterCursorState)
            + 4 // change count
        );
        try {
            for (var _c = __values(this.changes), _d = _c.next(); !_d.done; _d = _c.next()) {
                var change = _d.value;
                necessarySize += change.writeSize();
            }
        }
        catch (e_2_1) { e_2 = { error: e_2_1 }; }
        finally {
            try {
                if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
            }
            finally { if (e_2) throw e_2.error; }
        }
        var b = new Uint8Array(necessarySize);
        var offset = 0;
        buffer.writeUInt32BE(b, this.beforeVersionId, offset);
        offset += 4;
        buffer.writeUInt32BE(b, this.afterVersionId, offset);
        offset += 4;
        buffer.writeUInt8(b, this.beforeEOL, offset);
        offset += 1;
        buffer.writeUInt8(b, this.afterEOL, offset);
        offset += 1;
        offset = SingleModelEditStackData._writeSelections(b, this.beforeCursorState, offset);
        offset = SingleModelEditStackData._writeSelections(b, this.afterCursorState, offset);
        buffer.writeUInt32BE(b, this.changes.length, offset);
        offset += 4;
        try {
            for (var _e = __values(this.changes), _f = _e.next(); !_f.done; _f = _e.next()) {
                var change = _f.value;
                offset = change.write(b, offset);
            }
        }
        catch (e_3_1) { e_3 = { error: e_3_1 }; }
        finally {
            try {
                if (_f && !_f.done && (_b = _e.return)) _b.call(_e);
            }
            finally { if (e_3) throw e_3.error; }
        }
        return b.buffer;
    };
    SingleModelEditStackData.deserialize = function (source) {
        var b = new Uint8Array(source);
        var offset = 0;
        var beforeVersionId = buffer.readUInt32BE(b, offset);
        offset += 4;
        var afterVersionId = buffer.readUInt32BE(b, offset);
        offset += 4;
        var beforeEOL = buffer.readUInt8(b, offset);
        offset += 1;
        var afterEOL = buffer.readUInt8(b, offset);
        offset += 1;
        var beforeCursorState = [];
        offset = SingleModelEditStackData._readSelections(b, offset, beforeCursorState);
        var afterCursorState = [];
        offset = SingleModelEditStackData._readSelections(b, offset, afterCursorState);
        var changeCount = buffer.readUInt32BE(b, offset);
        offset += 4;
        var changes = [];
        for (var i = 0; i < changeCount; i++) {
            offset = TextChange.read(b, offset, changes);
        }
        return new SingleModelEditStackData(beforeVersionId, afterVersionId, beforeEOL, afterEOL, beforeCursorState, afterCursorState, changes);
    };
    return SingleModelEditStackData;
}());
var SingleModelEditStackElement = /** @class */ (function () {
    function SingleModelEditStackElement(model, beforeCursorState) {
        this.model = model;
        this._data = SingleModelEditStackData.create(model, beforeCursorState);
    }
    Object.defineProperty(SingleModelEditStackElement.prototype, "type", {
        get: function () {
            return 0 /* Resource */;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(SingleModelEditStackElement.prototype, "resource", {
        get: function () {
            if (URI.isUri(this.model)) {
                return this.model;
            }
            return this.model.uri;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(SingleModelEditStackElement.prototype, "label", {
        get: function () {
            return nls.localize('edit', "Typing");
        },
        enumerable: false,
        configurable: true
    });
    SingleModelEditStackElement.prototype.toString = function () {
        var data = (this._data instanceof SingleModelEditStackData ? this._data : SingleModelEditStackData.deserialize(this._data));
        return data.changes.map(function (change) { return change.toString(); }).join(', ');
    };
    SingleModelEditStackElement.prototype.matchesResource = function (resource) {
        var uri = (URI.isUri(this.model) ? this.model : this.model.uri);
        return (uri.toString() === resource.toString());
    };
    SingleModelEditStackElement.prototype.setModel = function (model) {
        this.model = model;
    };
    SingleModelEditStackElement.prototype.canAppend = function (model) {
        return (this.model === model && this._data instanceof SingleModelEditStackData);
    };
    SingleModelEditStackElement.prototype.append = function (model, textChanges, afterEOL, afterVersionId, afterCursorState) {
        if (this._data instanceof SingleModelEditStackData) {
            this._data.append(model, textChanges, afterEOL, afterVersionId, afterCursorState);
        }
    };
    SingleModelEditStackElement.prototype.close = function () {
        if (this._data instanceof SingleModelEditStackData) {
            this._data = this._data.serialize();
        }
    };
    SingleModelEditStackElement.prototype.open = function () {
        if (!(this._data instanceof SingleModelEditStackData)) {
            this._data = SingleModelEditStackData.deserialize(this._data);
        }
    };
    SingleModelEditStackElement.prototype.undo = function () {
        if (URI.isUri(this.model)) {
            // don't have a model
            throw new Error("Invalid SingleModelEditStackElement");
        }
        if (this._data instanceof SingleModelEditStackData) {
            this._data = this._data.serialize();
        }
        var data = SingleModelEditStackData.deserialize(this._data);
        this.model._applyUndo(data.changes, data.beforeEOL, data.beforeVersionId, data.beforeCursorState);
    };
    SingleModelEditStackElement.prototype.redo = function () {
        if (URI.isUri(this.model)) {
            // don't have a model
            throw new Error("Invalid SingleModelEditStackElement");
        }
        if (this._data instanceof SingleModelEditStackData) {
            this._data = this._data.serialize();
        }
        var data = SingleModelEditStackData.deserialize(this._data);
        this.model._applyRedo(data.changes, data.afterEOL, data.afterVersionId, data.afterCursorState);
    };
    SingleModelEditStackElement.prototype.heapSize = function () {
        if (this._data instanceof SingleModelEditStackData) {
            this._data = this._data.serialize();
        }
        return this._data.byteLength + 168 /*heap overhead*/;
    };
    return SingleModelEditStackElement;
}());
export { SingleModelEditStackElement };
var MultiModelEditStackElement = /** @class */ (function () {
    function MultiModelEditStackElement(label, editStackElements) {
        var e_4, _a;
        this.type = 1 /* Workspace */;
        this.label = label;
        this._isOpen = true;
        this._editStackElementsArr = editStackElements.slice(0);
        this._editStackElementsMap = new Map();
        try {
            for (var _b = __values(this._editStackElementsArr), _c = _b.next(); !_c.done; _c = _b.next()) {
                var editStackElement = _c.value;
                var key = uriGetComparisonKey(editStackElement.resource);
                this._editStackElementsMap.set(key, editStackElement);
            }
        }
        catch (e_4_1) { e_4 = { error: e_4_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_4) throw e_4.error; }
        }
        this._delegate = null;
    }
    Object.defineProperty(MultiModelEditStackElement.prototype, "resources", {
        get: function () {
            return this._editStackElementsArr.map(function (editStackElement) { return editStackElement.resource; });
        },
        enumerable: false,
        configurable: true
    });
    MultiModelEditStackElement.prototype.setDelegate = function (delegate) {
        this._delegate = delegate;
    };
    MultiModelEditStackElement.prototype.prepareUndoRedo = function () {
        if (this._delegate) {
            return this._delegate.prepareUndoRedo(this);
        }
    };
    MultiModelEditStackElement.prototype.getMissingModels = function () {
        var e_5, _a;
        var result = [];
        try {
            for (var _b = __values(this._editStackElementsArr), _c = _b.next(); !_c.done; _c = _b.next()) {
                var editStackElement = _c.value;
                if (URI.isUri(editStackElement.model)) {
                    result.push(editStackElement.model);
                }
            }
        }
        catch (e_5_1) { e_5 = { error: e_5_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_5) throw e_5.error; }
        }
        return result;
    };
    MultiModelEditStackElement.prototype.matchesResource = function (resource) {
        var key = uriGetComparisonKey(resource);
        return (this._editStackElementsMap.has(key));
    };
    MultiModelEditStackElement.prototype.setModel = function (model) {
        var key = uriGetComparisonKey(URI.isUri(model) ? model : model.uri);
        if (this._editStackElementsMap.has(key)) {
            this._editStackElementsMap.get(key).setModel(model);
        }
    };
    MultiModelEditStackElement.prototype.canAppend = function (model) {
        if (!this._isOpen) {
            return false;
        }
        var key = uriGetComparisonKey(model.uri);
        if (this._editStackElementsMap.has(key)) {
            var editStackElement = this._editStackElementsMap.get(key);
            return editStackElement.canAppend(model);
        }
        return false;
    };
    MultiModelEditStackElement.prototype.append = function (model, textChanges, afterEOL, afterVersionId, afterCursorState) {
        var key = uriGetComparisonKey(model.uri);
        var editStackElement = this._editStackElementsMap.get(key);
        editStackElement.append(model, textChanges, afterEOL, afterVersionId, afterCursorState);
    };
    MultiModelEditStackElement.prototype.close = function () {
        this._isOpen = false;
    };
    MultiModelEditStackElement.prototype.open = function () {
        // cannot reopen
    };
    MultiModelEditStackElement.prototype.undo = function () {
        var e_6, _a;
        this._isOpen = false;
        try {
            for (var _b = __values(this._editStackElementsArr), _c = _b.next(); !_c.done; _c = _b.next()) {
                var editStackElement = _c.value;
                editStackElement.undo();
            }
        }
        catch (e_6_1) { e_6 = { error: e_6_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_6) throw e_6.error; }
        }
    };
    MultiModelEditStackElement.prototype.redo = function () {
        var e_7, _a;
        try {
            for (var _b = __values(this._editStackElementsArr), _c = _b.next(); !_c.done; _c = _b.next()) {
                var editStackElement = _c.value;
                editStackElement.redo();
            }
        }
        catch (e_7_1) { e_7 = { error: e_7_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_7) throw e_7.error; }
        }
    };
    MultiModelEditStackElement.prototype.heapSize = function (resource) {
        var key = uriGetComparisonKey(resource);
        if (this._editStackElementsMap.has(key)) {
            var editStackElement = this._editStackElementsMap.get(key);
            return editStackElement.heapSize();
        }
        return 0;
    };
    MultiModelEditStackElement.prototype.split = function () {
        return this._editStackElementsArr;
    };
    MultiModelEditStackElement.prototype.toString = function () {
        var e_8, _a;
        var result = [];
        try {
            for (var _b = __values(this._editStackElementsArr), _c = _b.next(); !_c.done; _c = _b.next()) {
                var editStackElement = _c.value;
                result.push(basename(editStackElement.resource) + ": " + editStackElement);
            }
        }
        catch (e_8_1) { e_8 = { error: e_8_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_8) throw e_8.error; }
        }
        return "{" + result.join(', ') + "}";
    };
    return MultiModelEditStackElement;
}());
export { MultiModelEditStackElement };
function getModelEOL(model) {
    var eol = model.getEOL();
    if (eol === '\n') {
        return 0 /* LF */;
    }
    else {
        return 1 /* CRLF */;
    }
}
export function isEditStackElement(element) {
    if (!element) {
        return false;
    }
    return ((element instanceof SingleModelEditStackElement) || (element instanceof MultiModelEditStackElement));
}
var EditStack = /** @class */ (function () {
    function EditStack(model, undoRedoService) {
        this._model = model;
        this._undoRedoService = undoRedoService;
    }
    EditStack.prototype.pushStackElement = function () {
        var lastElement = this._undoRedoService.getLastElement(this._model.uri);
        if (isEditStackElement(lastElement)) {
            lastElement.close();
        }
    };
    EditStack.prototype.popStackElement = function () {
        var lastElement = this._undoRedoService.getLastElement(this._model.uri);
        if (isEditStackElement(lastElement)) {
            lastElement.open();
        }
    };
    EditStack.prototype.clear = function () {
        this._undoRedoService.removeElements(this._model.uri);
    };
    EditStack.prototype._getOrCreateEditStackElement = function (beforeCursorState) {
        var lastElement = this._undoRedoService.getLastElement(this._model.uri);
        if (isEditStackElement(lastElement) && lastElement.canAppend(this._model)) {
            return lastElement;
        }
        var newElement = new SingleModelEditStackElement(this._model, beforeCursorState);
        this._undoRedoService.pushElement(newElement);
        return newElement;
    };
    EditStack.prototype.pushEOL = function (eol) {
        var editStackElement = this._getOrCreateEditStackElement(null);
        this._model.setEOL(eol);
        editStackElement.append(this._model, [], getModelEOL(this._model), this._model.getAlternativeVersionId(), null);
    };
    EditStack.prototype.pushEditOperation = function (beforeCursorState, editOperations, cursorStateComputer) {
        var editStackElement = this._getOrCreateEditStackElement(beforeCursorState);
        var inverseEditOperations = this._model.applyEdits(editOperations, true);
        var afterCursorState = EditStack._computeCursorState(cursorStateComputer, inverseEditOperations);
        var textChanges = inverseEditOperations.map(function (op, index) { return ({ index: index, textChange: op.textChange }); });
        textChanges.sort(function (a, b) {
            if (a.textChange.oldPosition === b.textChange.oldPosition) {
                return a.index - b.index;
            }
            return a.textChange.oldPosition - b.textChange.oldPosition;
        });
        editStackElement.append(this._model, textChanges.map(function (op) { return op.textChange; }), getModelEOL(this._model), this._model.getAlternativeVersionId(), afterCursorState);
        return afterCursorState;
    };
    EditStack._computeCursorState = function (cursorStateComputer, inverseEditOperations) {
        try {
            return cursorStateComputer ? cursorStateComputer(inverseEditOperations) : null;
        }
        catch (e) {
            onUnexpectedError(e);
            return null;
        }
    };
    return EditStack;
}());
export { EditStack };
