"use strict";
/********************************************************************************
 * Copyright (C) 2018 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.electron = exports.ElectronFileDialogService = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const electron_1 = require("@theia/core/shared/electron");
const os_1 = require("@theia/core/lib/common/os");
const message_service_1 = require("@theia/core/lib/common/message-service");
const filesystem_1 = require("../../common/filesystem");
const file_dialog_1 = require("../../browser/file-dialog");
//
// We are OK to use this here because the electron backend and frontend are on the same host.
// If required, we can move this single service (and its module) to a dedicated Theia extension,
// and at packaging time, clients can decide whether they need the native or the browser-based
// solution.
//
const file_uri_1 = require("@theia/core/lib/node/file-uri");
let ElectronFileDialogService = class ElectronFileDialogService extends file_dialog_1.DefaultFileDialogService {
    async showOpenDialog(props, folder) {
        const rootNode = await this.getRootNode(folder);
        if (rootNode) {
            const { filePaths } = await electron_1.remote.dialog.showOpenDialog(this.toOpenDialogOptions(rootNode.uri, props));
            if (filePaths.length === 0) {
                return undefined;
            }
            const uris = filePaths.map(path => file_uri_1.FileUri.create(path));
            const canAccess = await this.canRead(uris);
            const result = canAccess ? uris.length === 1 ? uris[0] : uris : undefined;
            return result;
        }
        return undefined;
    }
    async showSaveDialog(props, folder) {
        const rootNode = await this.getRootNode(folder);
        if (rootNode) {
            const { filePath } = await electron_1.remote.dialog.showSaveDialog(this.toSaveDialogOptions(rootNode.uri, props));
            if (!filePath) {
                return undefined;
            }
            const uri = file_uri_1.FileUri.create(filePath);
            const exists = await this.fileService.exists(uri);
            if (!exists) {
                return uri;
            }
            const canWrite = await this.canReadWrite(uri);
            return canWrite ? uri : undefined;
        }
        return undefined;
    }
    async canReadWrite(uris) {
        for (const uri of Array.isArray(uris) ? uris : [uris]) {
            if (!(await this.fileService.access(uri, filesystem_1.FileAccess.Constants.R_OK | filesystem_1.FileAccess.Constants.W_OK))) {
                this.messageService.error(`Cannot access resource at ${uri.path}.`);
                return false;
            }
        }
        return true;
    }
    async canRead(uris) {
        const inaccessibleFilePaths = await Promise.all((Array.isArray(uris) ? uris : [uris]).map(async (uri) => (!await this.fileService.access(uri, filesystem_1.FileAccess.Constants.R_OK) && uri.path || '')).filter(e => e));
        if (inaccessibleFilePaths.length) {
            this.messageService.error(`Cannot read ${inaccessibleFilePaths.length} resources: ${inaccessibleFilePaths.join(', ')}`);
        }
        return !!inaccessibleFilePaths.length;
    }
    toDialogOptions(uri, props, dialogTitle) {
        const title = props.title || dialogTitle;
        const defaultPath = file_uri_1.FileUri.fsPath(uri);
        const filters = [{ name: 'All Files', extensions: ['*'] }];
        if (props.filters) {
            filters.unshift(...Object.keys(props.filters).map(key => ({ name: key, extensions: props.filters[key] })));
        }
        return { title, defaultPath, filters };
    }
    toOpenDialogOptions(uri, props) {
        const properties = electron.dialog.toDialogProperties(props);
        const buttonLabel = props.openLabel;
        return Object.assign(Object.assign({}, this.toDialogOptions(uri, props, 'Open')), { properties, buttonLabel });
    }
    toSaveDialogOptions(uri, props) {
        const buttonLabel = props.saveLabel;
        if (props.inputValue) {
            uri = uri.resolve(props.inputValue);
        }
        const defaultPath = file_uri_1.FileUri.fsPath(uri);
        return Object.assign(Object.assign({}, this.toDialogOptions(uri, props, 'Save')), { buttonLabel, defaultPath });
    }
};
__decorate([
    inversify_1.inject(message_service_1.MessageService),
    __metadata("design:type", message_service_1.MessageService)
], ElectronFileDialogService.prototype, "messageService", void 0);
ElectronFileDialogService = __decorate([
    inversify_1.injectable()
], ElectronFileDialogService);
exports.ElectronFileDialogService = ElectronFileDialogService;
var electron;
(function (electron) {
    let dialog;
    (function (dialog) {
        /**
         * Converts the Theia specific `OpenFileDialogProps` into an electron specific array.
         *
         * Note: On Windows and Linux an open dialog can not be both a file selector and a directory selector,
         * so if you set properties to ['openFile', 'openDirectory'] on these platforms, a directory selector will be shown.
         *
         * See: https://github.com/electron/electron/issues/10252#issuecomment-322012159
         */
        function toDialogProperties(props) {
            if (!os_1.isOSX && props.canSelectFiles !== false && props.canSelectFolders === true) {
                console.warn(`Cannot have 'canSelectFiles' and 'canSelectFolders' at the same time. Fallback to 'folder' dialog. \nProps was: ${JSON.stringify(props)}.`);
                // Given that both props are set, fallback to using a `folder` dialog.
                props.canSelectFiles = false;
                props.canSelectFolders = true;
            }
            const properties = [];
            if (!os_1.isOSX) {
                if (props.canSelectFiles !== false && props.canSelectFolders !== true) {
                    properties.push('openFile');
                }
                if (props.canSelectFolders === true && props.canSelectFiles === false) {
                    properties.push('openDirectory');
                }
            }
            else {
                if (props.canSelectFiles !== false) {
                    properties.push('openFile');
                }
                if (props.canSelectFolders === true) {
                    properties.push('openDirectory');
                    properties.push('createDirectory');
                }
            }
            if (props.canSelectMany === true) {
                properties.push('multiSelections');
            }
            return properties;
        }
        dialog.toDialogProperties = toDialogProperties;
    })(dialog = electron.dialog || (electron.dialog = {}));
})(electron = exports.electron || (exports.electron = {}));
//# sourceMappingURL=electron-file-dialog-service.js.map