"use strict";
/********************************************************************************
 * Copyright (C) 2017 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ElectronMenuContribution = exports.ElectronMenus = exports.ElectronCommands = void 0;
const electron = require("../../../shared/electron");
const inversify_1 = require("inversify");
const common_1 = require("../../common");
const browser_1 = require("../../browser");
const browser_2 = require("../../browser");
const electron_main_menu_factory_1 = require("./electron-main-menu-factory");
const frontend_application_state_1 = require("../../browser/frontend-application-state");
const electron_window_preferences_1 = require("../window/electron-window-preferences");
var ElectronCommands;
(function (ElectronCommands) {
    ElectronCommands.TOGGLE_DEVELOPER_TOOLS = {
        id: 'theia.toggleDevTools',
        label: 'Toggle Developer Tools'
    };
    ElectronCommands.RELOAD = {
        id: 'view.reload',
        label: 'Reload Window'
    };
    ElectronCommands.ZOOM_IN = {
        id: 'view.zoomIn',
        label: 'Zoom In'
    };
    ElectronCommands.ZOOM_OUT = {
        id: 'view.zoomOut',
        label: 'Zoom Out'
    };
    ElectronCommands.RESET_ZOOM = {
        id: 'view.resetZoom',
        label: 'Reset Zoom'
    };
    ElectronCommands.CLOSE_WINDOW = {
        id: 'close.window',
        label: 'Close Window'
    };
    ElectronCommands.TOGGLE_FULL_SCREEN = {
        id: 'workbench.action.toggleFullScreen',
        category: 'View',
        label: 'Toggle Full Screen'
    };
})(ElectronCommands = exports.ElectronCommands || (exports.ElectronCommands = {}));
var ElectronMenus;
(function (ElectronMenus) {
    ElectronMenus.VIEW_WINDOW = [...browser_2.CommonMenus.VIEW, 'window'];
    ElectronMenus.VIEW_ZOOM = [...browser_2.CommonMenus.VIEW, 'zoom'];
})(ElectronMenus = exports.ElectronMenus || (exports.ElectronMenus = {}));
(function (ElectronMenus) {
    ElectronMenus.HELP_TOGGLE = [...browser_2.CommonMenus.HELP, 'z_toggle'];
})(ElectronMenus = exports.ElectronMenus || (exports.ElectronMenus = {}));
(function (ElectronMenus) {
    ElectronMenus.FILE_CLOSE = [...browser_2.CommonMenus.FILE_CLOSE, 'window-close'];
})(ElectronMenus = exports.ElectronMenus || (exports.ElectronMenus = {}));
let ElectronMenuContribution = class ElectronMenuContribution {
    constructor(factory, shell) {
        this.factory = factory;
        this.shell = shell;
    }
    onStart(app) {
        this.hideTopPanel(app);
        this.preferenceService.ready.then(() => {
            this.setMenu();
            electron.remote.getCurrentWindow().setMenuBarVisibility(true);
        });
        if (common_1.isOSX) {
            // OSX: Recreate the menus when changing windows.
            // OSX only has one menu bar for all windows, so we need to swap
            // between them as the user switches windows.
            electron.remote.getCurrentWindow().on('focus', () => this.setMenu());
        }
        // Make sure the application menu is complete, once the frontend application is ready.
        // https://github.com/theia-ide/theia/issues/5100
        let onStateChange = undefined;
        const stateServiceListener = (state) => {
            if (state === 'ready') {
                this.setMenu();
            }
            if (state === 'closing_window') {
                if (!!onStateChange) {
                    onStateChange.dispose();
                }
            }
        };
        onStateChange = this.stateService.onStateChanged(stateServiceListener);
        this.shell.mainPanel.onDidToggleMaximized(() => {
            this.handleToggleMaximized();
        });
        this.shell.bottomPanel.onDidToggleMaximized(() => {
            this.handleToggleMaximized();
        });
    }
    handleToggleMaximized() {
        const preference = this.preferenceService.get('window.menuBarVisibility');
        if (preference === 'classic') {
            this.factory.setMenuBar();
        }
    }
    /**
     * Makes the `theia-top-panel` hidden as it is unused for the electron-based application.
     * The `theia-top-panel` is used as the container of the main, application menu-bar for the
     * browser. Electron has it's own.
     * By default, this method is called on application `onStart`.
     */
    hideTopPanel(app) {
        const itr = app.shell.children();
        let child = itr.next();
        while (child) {
            // Top panel for the menu contribution is not required for Electron.
            if (child.id === 'theia-top-panel') {
                child.setHidden(true);
                child = undefined;
            }
            else {
                child = itr.next();
            }
        }
    }
    setMenu(menu = this.factory.createMenuBar(), electronWindow = electron.remote.getCurrentWindow()) {
        if (common_1.isOSX) {
            electron.remote.Menu.setApplicationMenu(menu);
        }
        else {
            // Unix/Windows: Set the per-window menus
            electronWindow.setMenu(menu);
        }
    }
    registerCommands(registry) {
        const currentWindow = electron.remote.getCurrentWindow();
        registry.registerCommand(ElectronCommands.TOGGLE_DEVELOPER_TOOLS, {
            execute: () => {
                const webContent = electron.remote.getCurrentWebContents();
                if (!webContent.isDevToolsOpened()) {
                    webContent.openDevTools();
                }
                else {
                    webContent.closeDevTools();
                }
            }
        });
        registry.registerCommand(ElectronCommands.RELOAD, {
            execute: () => currentWindow.reload()
        });
        registry.registerCommand(ElectronCommands.CLOSE_WINDOW, {
            execute: () => currentWindow.close()
        });
        registry.registerCommand(ElectronCommands.ZOOM_IN, {
            execute: () => {
                const webContents = currentWindow.webContents;
                // When starting at a level that is not a multiple of 0.5, increment by at most 0.5 to reach the next highest multiple of 0.5.
                let zoomLevel = (Math.floor(webContents.zoomLevel / electron_window_preferences_1.ZoomLevel.VARIATION) * electron_window_preferences_1.ZoomLevel.VARIATION) + electron_window_preferences_1.ZoomLevel.VARIATION;
                if (zoomLevel > electron_window_preferences_1.ZoomLevel.MAX) {
                    zoomLevel = electron_window_preferences_1.ZoomLevel.MAX;
                    return;
                }
                ;
                this.preferenceService.set('window.zoomLevel', zoomLevel, browser_1.PreferenceScope.User);
            }
        });
        registry.registerCommand(ElectronCommands.ZOOM_OUT, {
            execute: () => {
                const webContents = currentWindow.webContents;
                // When starting at a level that is not a multiple of 0.5, decrement by at most 0.5 to reach the next lowest multiple of 0.5.
                let zoomLevel = (Math.ceil(webContents.zoomLevel / electron_window_preferences_1.ZoomLevel.VARIATION) * electron_window_preferences_1.ZoomLevel.VARIATION) - electron_window_preferences_1.ZoomLevel.VARIATION;
                if (zoomLevel < electron_window_preferences_1.ZoomLevel.MIN) {
                    zoomLevel = electron_window_preferences_1.ZoomLevel.MIN;
                    return;
                }
                ;
                this.preferenceService.set('window.zoomLevel', zoomLevel, browser_1.PreferenceScope.User);
            }
        });
        registry.registerCommand(ElectronCommands.RESET_ZOOM, {
            execute: () => this.preferenceService.set('window.zoomLevel', electron_window_preferences_1.ZoomLevel.DEFAULT, browser_1.PreferenceScope.User)
        });
        registry.registerCommand(ElectronCommands.TOGGLE_FULL_SCREEN, {
            isEnabled: () => currentWindow.isFullScreenable(),
            isVisible: () => currentWindow.isFullScreenable(),
            execute: () => currentWindow.setFullScreen(!currentWindow.isFullScreen())
        });
    }
    registerKeybindings(registry) {
        registry.registerKeybindings({
            command: ElectronCommands.TOGGLE_DEVELOPER_TOOLS.id,
            keybinding: 'ctrlcmd+alt+i'
        }, {
            command: ElectronCommands.RELOAD.id,
            keybinding: 'ctrlcmd+r'
        }, {
            command: ElectronCommands.ZOOM_IN.id,
            keybinding: 'ctrlcmd+='
        }, {
            command: ElectronCommands.ZOOM_OUT.id,
            keybinding: 'ctrlcmd+-'
        }, {
            command: ElectronCommands.RESET_ZOOM.id,
            keybinding: 'ctrlcmd+0'
        }, {
            command: ElectronCommands.CLOSE_WINDOW.id,
            keybinding: (common_1.isOSX ? 'cmd+shift+w' : (common_1.isWindows ? 'ctrl+w' : /* Linux */ 'ctrl+q'))
        }, {
            command: ElectronCommands.TOGGLE_FULL_SCREEN.id,
            keybinding: common_1.isOSX ? 'ctrl+ctrlcmd+f' : 'f11'
        });
    }
    registerMenus(registry) {
        registry.registerMenuAction(ElectronMenus.HELP_TOGGLE, {
            commandId: ElectronCommands.TOGGLE_DEVELOPER_TOOLS.id
        });
        registry.registerMenuAction(ElectronMenus.VIEW_WINDOW, {
            commandId: ElectronCommands.RELOAD.id,
            order: 'z0'
        });
        registry.registerMenuAction(ElectronMenus.VIEW_ZOOM, {
            commandId: ElectronCommands.ZOOM_IN.id,
            order: 'z1'
        });
        registry.registerMenuAction(ElectronMenus.VIEW_ZOOM, {
            commandId: ElectronCommands.ZOOM_OUT.id,
            order: 'z2'
        });
        registry.registerMenuAction(ElectronMenus.VIEW_ZOOM, {
            commandId: ElectronCommands.RESET_ZOOM.id,
            order: 'z3'
        });
        registry.registerMenuAction(ElectronMenus.FILE_CLOSE, {
            commandId: ElectronCommands.CLOSE_WINDOW.id,
        });
    }
};
__decorate([
    inversify_1.inject(frontend_application_state_1.FrontendApplicationStateService),
    __metadata("design:type", frontend_application_state_1.FrontendApplicationStateService)
], ElectronMenuContribution.prototype, "stateService", void 0);
__decorate([
    inversify_1.inject(browser_1.PreferenceService),
    __metadata("design:type", Object)
], ElectronMenuContribution.prototype, "preferenceService", void 0);
ElectronMenuContribution = __decorate([
    inversify_1.injectable(),
    __param(0, inversify_1.inject(electron_main_menu_factory_1.ElectronMainMenuFactory)),
    __param(1, inversify_1.inject(browser_1.ApplicationShell)),
    __metadata("design:paramtypes", [electron_main_menu_factory_1.ElectronMainMenuFactory,
        browser_1.ApplicationShell])
], ElectronMenuContribution);
exports.ElectronMenuContribution = ElectronMenuContribution;
//# sourceMappingURL=electron-menu-contribution.js.map