# Commands covered: expr
#
# This file contains a collection of tests for one or more of the Tcl
# built-in commands. Sourcing this file into Tcl runs the tests and
# generates output for errors.  No output means no errors were found.
#
# Copyright (c) 1996-1997 Sun Microsystems, Inc.
# Copyright (c) 1998-2000 by Scriptics Corporation.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# RCS: @(#) $Id: expr.tcl,v 1.1 2006/12/29 09:02:53 fourdman Exp $

if {[lsearch [namespace children] ::tcltest] == -1} {
    package require tcltest
    namespace import -force ::tcltest::*
}

testConstraint registeredMathFuncs [expr {
    ([catch {expr T1()} msg] != 1) || ($msg ne {unknown math function "T1"})
}]

# procedures used below

proc put_hello_char {c} {
    global a
    append a [format %c $c]
    return $c
}
proc hello_world {} {
    global a
    set a ""
    set L1 [set l0 [set h_1 [set q 0]]]
    for {put_hello_char [expr [put_hello_char [expr [set h 7]*10+2]]+29]} {$l0?[put_hello_char $l0]
        :!$h_1} {put_hello_char $ll;expr {$L1==2?[set ll [expr 32+0-0+[set bar 0]]]:0}} {expr {[incr L1]==[expr 1+([string length "abc"]-[string length "abc"])]
        ?[set ll [set l0 [expr 54<<1]]]:$ll==108&&$L1<3?
        [incr ll [expr 1|1<<1]; set ll $ll; set ll $ll; set ll $ll; set ll $ll; set l0 [expr ([string length "abc"]-[string length "abc"])+([string length "abc"]-[string length "abc"])-([string length "abc"]-[string length "abc"])+([string length "abc"]-[string length "abc"])]; set l0; set l0 $l0; set l0; set l0]:$L1==4&&$ll==32?[set ll [expr 19+$h1+([string length "abc"]-[string length "abc"])-([string length "abc"]-[string length "abc"])+([string length "abc"]-[string length "abc"])-([string length "abc"]-[string length "abc"])+[set foo [expr ([string length "abc"]-[string length "abc"])+([string length "abc"]-[string length "abc"])+([string length "abc"]-[string length "abc"])]]]]
        :[set q [expr $q-$h1+([string length "abc"]-[string length "abc"])-([string length "abc"]-[string length "abc"])]]};expr {$L1==5?[incr ll -8; set ll $ll; set ll]:$q&&$h1&&1};expr {$L1==4+2
        ?[incr ll 3]:[expr ([string length "abc"]-[string length "abc"])+1]};expr {$ll==($h<<4)+2+0&&$L1!=6?[incr ll -6]:[set h1 [expr 100+([string length "abc"]-[string length "abc"])-([string length "abc"]-[string length "abc"])]]}
        expr {$L1!=1<<3?[incr q [expr ([string length "abc"]-[string length "abc"])-1]]:[set h_1 [set ll $h1]]}
    }
    set a
}

proc 12days {a b c} {
    global xxx
    expr {1<$a?[expr {$a<3?[12days -79 -13 [string range $c [12days -87 \
	[expr 1-$b] [string range $c [12days -86 0 [string range $c 1 end]] \
	end]] end]]:1};expr {$a<$b?[12days [expr $a+1] $b $c]:3};expr {[12days \
	-94 [expr $a-27] $c]&&$a==2?$b<13?[12days 2 [expr $b+1] "%s %d %d\n"]:9
	:16}]:$a<0?$a<-72?[12days $b $a "@n'+,#'/*\{\}w+/w#cdnr/+,\{\}r/*de\}+,/*\{*+,/w\{%+,/w#q#n+,/#\{l+,/n\{n+,/+#n+,/#;#q#n+,/+k#;*+,/'r :'d*'3,\}\{w+K w'K:'+\}e#';dq#'l q#'+d'K#!/+k#;q#'r\}eKK#\}w'r\}eKK\{nl\]'/#;#q#n')\{)#\}w')\{)\{nl\]'/+#n';d\}rw' i;# )\{nl\]!/n\{n#'; r\{#w'r nc\{nl\]'/#\{l,+'K \{rw' iK\{;\[\{nl\]'/w#q#n'wk nw' iwk\{KK\{nl\]!/w\{%'l##w#' i; :\{nl\]'/*\{q#'ld;r'\}\{nlwb!/*de\}'c ;;\{nl'-\{\}rw\]'/+,\}##'*\}#nc,',#nw\]'/+kd'+e\}+;#'rdq#w! nr'/ ') \}+\}\{rl#'\{n' ')# \}'+\}##(!!/"]
	:$a<-50?[string compare [format %c $b] [string index $c 0]]==0?[append \
	xxx [string index $c 31];scan [string index $c 31] %c x;set x]
	:[12days -65 $b [string range $c 1 end]]:[12days [expr ([string compare \
	[string index $c 0] "/"]==0)+$a] $b [string range $c 1 end]]:0<$a
	?[12days 2 2 "%s"]:[string compare [string index $c 0] "/"]==0||
	[12days 0 [12days -61 [scan [string index $c 0] %c x; set x] \
	"!ek;dc i@bK'(q)-\[w\]*%n+r3#l,\{\}:\nuwloca-O;m .vpbks,fxntdCeghiry"] \
	[string range $c 1 end]]}
}
proc do_twelve_days {} {
    global xxx
    set xxx ""
    12days 1 1 1
    string length $xxx
}

# start of tests

catch {unset a b i x}

test expr-1.1 {TclCompileExprCmd: no expression} {
    list [catch {expr  } msg] $msg
} {1 {wrong # args: should be "expr arg ?arg ...?"}}
test expr-1.2 {TclCompileExprCmd: one expression word} {
    expr -25
} -25
test expr-1.3 {TclCompileExprCmd: two expression words} {
    expr -8.2   -6
} -14.2
test expr-1.4 {TclCompileExprCmd: five expression words} {
    expr 20 - 5 +10 -7
} 18
test expr-1.5 {TclCompileExprCmd: quoted expression word} {
    expr "0005"
} 5
test expr-1.6 {TclCompileExprCmd: quoted expression word} {
    catch {expr "0005"zxy} msg
    set msg
} {extra characters after close-quote}
test expr-1.7 {TclCompileExprCmd: expression word in braces} {
    expr {-0005}
} -5
test expr-1.8 {TclCompileExprCmd: expression word in braces} {
    expr {{-0x1234}}
} -4660
test expr-1.9 {TclCompileExprCmd: expression word in braces} {
    catch {expr {-0005}foo} msg
    set msg
} {extra characters after close-brace}
test expr-1.10 {TclCompileExprCmd: other expression word in braces} {
    expr 4*[llength "6 2"]
} 8
test expr-1.11 {TclCompileExprCmd: expression word terminated by ;} {
    expr 4*[llength "6 2"];
} 8
test expr-1.12 {TclCompileExprCmd: inlined expr (in "catch") inside other catch} {
    set a xxx
    catch {
	# Might not be a number
	set a [expr 10*$a]
    }
} 1
test expr-1.13 {TclCompileExprCmd: second level of substitutions in expr not in braces with single var reference} {
    set a xxx
    set x 27;  set bool {$x};  if $bool {set a foo}
    set a
} foo
test expr-1.14 {TclCompileExprCmd: second level of substitutions in expr with comparison as top-level operator} {
    set a xxx
    set x 2;  set b {$x};  set a [expr $b == 2]
    set a
} 1
test expr-1.15 {TclCompileExprCmd: second level of substitutions in expr with comparison as top-level operator} {
    set a xxx
    set x 2;  set b {$x};  set a [expr $b eq 2]
    set a
} 1

test expr-2.1 {TclCompileExpr: are builtin functions registered?} {
    expr double(5*[llength "6 2"])
} 10.0
test expr-2.2 {TclCompileExpr: error in expr} {
    catch {expr 2**3} msg
    set msg
} {syntax error in expression "2**3": unexpected operator *}
test expr-2.3 {TclCompileExpr: junk after legal expr} {
    catch {expr 7*[llength "a b"]foo} msg
    set msg
} {syntax error in expression "7*2foo": extra tokens at end of expression}
test expr-2.4 {TclCompileExpr: numeric expr string rep == formatted int rep} {
    expr {0001}
} 1

test expr-3.1 {CompileCondExpr: just lor expr} {expr 3||0} 1
test expr-3.2 {CompileCondExpr: error in lor expr} {
    catch {expr x||3} msg
    set msg
} {syntax error in expression "x||3": variable references require preceding $}
test expr-3.3 {CompileCondExpr: test true arm} {expr 3>2?44:66} 44
test expr-3.4 {CompileCondExpr: error compiling true arm} {
    catch {expr 3>2?2**3:66} msg
    set msg
} {syntax error in expression "3>2?2**3:66": unexpected operator *}
test expr-3.5 {CompileCondExpr: test false arm} {expr 2>3?44:66} 66
test expr-3.6 {CompileCondExpr: error compiling false arm} {
    catch {expr 2>3?44:2**3} msg
    set msg
} {syntax error in expression "2>3?44:2**3": unexpected operator *}
test expr-3.7 {CompileCondExpr: long arms & nested cond exprs} {unixOnly nonPortable} {
    puts "Note: doing test expr-3.7 which can take several minutes to run"
    hello_world
} {Hello world}
catch {unset xxx}
test expr-3.8 {CompileCondExpr: long arms & nested cond exprs} {unixOnly nonPortable} {
    puts "Note: doing test expr-3.8 which can take several minutes to run"
    do_twelve_days
} 2358
catch {unset xxx}

test expr-4.1 {CompileLorExpr: just land expr} {expr 1.3&&3.3} 1
test expr-4.2 {CompileLorExpr: error in land expr} {
    catch {expr x&&3} msg
    set msg
} {syntax error in expression "x&&3": variable references require preceding $} 
test expr-4.3 {CompileLorExpr: simple lor exprs} {expr 0||1.0} 1
test expr-4.4 {CompileLorExpr: simple lor exprs} {expr 3.0||0.0} 1
test expr-4.5 {CompileLorExpr: simple lor exprs} {expr 0||0||1} 1
test expr-4.6 {CompileLorExpr: error compiling lor arm} {
    catch {expr 2**3||4.0} msg
    set msg
} {syntax error in expression "2**3||4.0": unexpected operator *}
test expr-4.7 {CompileLorExpr: error compiling lor arm} {
    catch {expr 1.3||2**3} msg
    set msg
} {syntax error in expression "1.3||2**3": unexpected operator *}
test expr-4.8 {CompileLorExpr: error compiling lor arms} {
    list [catch {expr {"a"||"b"}} msg] $msg
} {1 {expected boolean value but got "a"}}
test expr-4.9 {CompileLorExpr: long lor arm} {
    set a "abcdefghijkl"
    set i 7
    expr {[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]] || [string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]] || [string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]] || [string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]}
} 1

test expr-5.1 {CompileLandExpr: just bitor expr} {expr 7|0x13} 23
test expr-5.2 {CompileLandExpr: error in bitor expr} {
    catch {expr x|3} msg
    set msg
} {syntax error in expression "x|3": variable references require preceding $} 
test expr-5.3 {CompileLandExpr: simple land exprs} {expr 0&&1.0} 0
test expr-5.4 {CompileLandExpr: simple land exprs} {expr 0&&0} 0
test expr-5.5 {CompileLandExpr: simple land exprs} {expr 3.0&&1.2} 1
test expr-5.6 {CompileLandExpr: simple land exprs} {expr 1&&1&&2} 1
test expr-5.7 {CompileLandExpr: error compiling land arm} {
    catch {expr 2**3&&4.0} msg
    set msg
} {syntax error in expression "2**3&&4.0": unexpected operator *}
test expr-5.8 {CompileLandExpr: error compiling land arm} {
    catch {expr 1.3&&2**3} msg
    set msg
} {syntax error in expression "1.3&&2**3": unexpected operator *}
test expr-5.9 {CompileLandExpr: error compiling land arm} {
    list [catch {expr {"a"&&"b"}} msg] $msg
} {1 {expected boolean value but got "a"}}
test expr-5.10 {CompileLandExpr: long land arms} {
    set a "abcdefghijkl"
    set i 7
    expr {[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]] && [string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]] && [string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]] && [string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]^[string compare [format %c 103] [string index $a $i]]^[string compare [format %c 105] [string index $a $i]]}
} 1

test expr-6.1 {CompileBitXorExpr: just bitand expr} {expr 7&0x13} 3
test expr-6.2 {CompileBitXorExpr: error in bitand expr} {
    catch {expr x|3} msg
    set msg
} {syntax error in expression "x|3": variable references require preceding $} 
test expr-6.3 {CompileBitXorExpr: simple bitxor exprs} {expr 7^0x13} 20
test expr-6.4 {CompileBitXorExpr: simple bitxor exprs} {expr 3^0x10} 19
test expr-6.5 {CompileBitXorExpr: simple bitxor exprs} {expr 0^7} 7
test expr-6.6 {CompileBitXorExpr: simple bitxor exprs} {expr -1^7} -8
test expr-6.7 {CompileBitXorExpr: error compiling bitxor arm} {
    catch {expr 2**3|6} msg
    set msg
} {syntax error in expression "2**3|6": unexpected operator *}
test expr-6.8 {CompileBitXorExpr: error compiling bitxor arm} {
    catch {expr 2^x} msg
    set msg
} {syntax error in expression "2^x": variable references require preceding $}
test expr-6.9 {CompileBitXorExpr: runtime error in bitxor arm} {
    list [catch {expr {24.0^3}} msg] $msg
} {1 {can't use floating-point value as operand of "^"}}
test expr-6.10 {CompileBitXorExpr: runtime error in bitxor arm} {
    list [catch {expr {"a"^"b"}} msg] $msg
} {1 {can't use non-numeric string as operand of "^"}}

test expr-7.1 {CompileBitAndExpr: just equality expr} {expr 3==2} 0
test expr-7.2 {CompileBitAndExpr: just equality expr} {expr 2.0==2} 1
test expr-7.3 {CompileBitAndExpr: just equality expr} {expr 3.2!=2.2} 1
test expr-7.4 {CompileBitAndExpr: just equality expr} {expr {"abc" == "abd"}} 0
test expr-7.5 {CompileBitAndExpr: error in equality expr} {
    catch {expr x==3} msg
    set msg
} {syntax error in expression "x==3": variable references require preceding $}
test expr-7.6 {CompileBitAndExpr: simple bitand exprs} {expr 7&0x13} 3
test expr-7.7 {CompileBitAndExpr: simple bitand exprs} {expr 0xf2&0x53} 82
test expr-7.8 {CompileBitAndExpr: simple bitand exprs} {expr 3&6} 2
test expr-7.9 {CompileBitAndExpr: simple bitand exprs} {expr -1&-7} -7
test expr-7.10 {CompileBitAndExpr: error compiling bitand arm} {
    catch {expr 2**3&6} msg
    set msg
} {syntax error in expression "2**3&6": unexpected operator *}
test expr-7.11 {CompileBitAndExpr: error compiling bitand arm} {
    catch {expr 2&x} msg
    set msg
} {syntax error in expression "2&x": variable references require preceding $}
test expr-7.12 {CompileBitAndExpr: runtime error in bitand arm} {
    list [catch {expr {24.0&3}} msg] $msg
} {1 {can't use floating-point value as operand of "&"}}
test expr-7.13 {CompileBitAndExpr: runtime error in bitand arm} {
    list [catch {expr {"a"&"b"}} msg] $msg
} {1 {can't use non-numeric string as operand of "&"}}
test expr-7.14 {CompileBitAndExpr: equality expr} {expr 3eq2} 0
test expr-7.18 {CompileBitAndExpr: equality expr} {expr {"abc" eq "abd"}} 0
test expr-7.20 {CompileBitAndExpr: error in equality expr} {
    catch {expr xne3} msg
    set msg
} {syntax error in expression "xne3": variable references require preceding $} 

test expr-8.1 {CompileEqualityExpr: just relational expr} {expr 3>=2} 1
test expr-8.2 {CompileEqualityExpr: just relational expr} {expr 2<=2.1} 1
test expr-8.3 {CompileEqualityExpr: just relational expr} {expr 3.2>"2.2"} 1
test expr-8.4 {CompileEqualityExpr: just relational expr} {expr {"0y"<"0x12"}} 0
test expr-8.5 {CompileEqualityExpr: error in relational expr} {
    catch {expr x>3} msg
    set msg
} {syntax error in expression "x>3": variable references require preceding $}
test expr-8.6 {CompileEqualityExpr: simple equality exprs} {expr 7==0x13} 0
test expr-8.7 {CompileEqualityExpr: simple equality exprs} {expr -0xf2!=0x53} 1
test expr-8.8 {CompileEqualityExpr: simple equality exprs} {expr {"12398712938788234-1298379" != ""}} 1
test expr-8.9 {CompileEqualityExpr: simple equality exprs} {expr -1!="abc"} 1
test expr-8.10 {CompileEqualityExpr: error compiling equality arm} {
    catch {expr 2**3==6} msg
    set msg
} {syntax error in expression "2**3==6": unexpected operator *}
test expr-8.11 {CompileEqualityExpr: error compiling equality arm} {
    catch {expr 2!=x} msg
    set msg
} {syntax error in expression "2!=x": variable references require preceding $}
test expr-8.12 {CompileBitAndExpr: equality expr} {expr {"a"eq"a"}} 1
test expr-8.13 {CompileBitAndExpr: equality expr} {expr {"\374" eq ""}} 1
test expr-8.14 {CompileBitAndExpr: equality expr} {expr 3eq2} 0
test expr-8.15 {CompileBitAndExpr: equality expr} {expr 2.0eq2} 0
test expr-8.16 {CompileBitAndExpr: equality expr} {expr 3.2ne2.2} 1
test expr-8.17 {CompileBitAndExpr: equality expr} {expr 01eq1} 0
test expr-8.18 {CompileBitAndExpr: equality expr} {expr {"abc" eq "abd"}} 0
test expr-8.19 {CompileBitAndExpr: equality expr} {expr {"abc" ne "abd"}} 1
test expr-8.20 {CompileBitAndExpr: error in equality expr} {
    catch {expr x ne3} msg
    set msg
} {syntax error in expression "x ne3": variable references require preceding $} 
test expr-8.21 {CompileBitAndExpr: error in equality expr} {
    # These should be ""ed to avoid the error
    catch {expr a eq b} msg
    set msg
} {syntax error in expression "a eq b": variable references require preceding $}

test expr-9.1 {CompileRelationalExpr: just shift expr} {expr 3<<2} 12
test expr-9.2 {CompileRelationalExpr: just shift expr} {expr 0xff>>2} 63
test expr-9.3 {CompileRelationalExpr: just shift expr} {expr -1>>2} -1
test expr-9.4 {CompileRelationalExpr: just shift expr} {expr {1<<3}} 8

# The following test is different for 32-bit versus 64-bit
# architectures because LONG_MIN is different

if {0x80000000 > 0} {
    test expr-9.5 {CompileRelationalExpr: shift expr producing LONG_MIN} {nonPortable} {
	expr {1<<63}
    } -9223372036854775808
} else {
    test expr-9.5 {CompileRelationalExpr: shift expr producing LONG_MIN} {nonPortable} {
	expr {1<<31}
    } -2147483648
}
test expr-9.6 {CompileRelationalExpr: error in shift expr} {
    catch {expr x>>3} msg
    set msg
} {syntax error in expression "x>>3": variable references require preceding $}
test expr-9.7 {CompileRelationalExpr: simple relational exprs} {expr 0xff>=+0x3} 1
test expr-9.8 {CompileRelationalExpr: simple relational exprs} {expr -0xf2<0x3} 1
test expr-9.9 {CompileRelationalExpr: error compiling relational arm} {
    catch {expr 2**3>6} msg
    set msg
} {syntax error in expression "2**3>6": unexpected operator *}
test expr-9.10 {CompileRelationalExpr: error compiling relational arm} {
    catch {expr 2<x} msg
    set msg
} {syntax error in expression "2<x": variable references require preceding $}

test expr-10.1 {CompileShiftExpr: just add expr} {expr 4+-2} 2
test expr-10.2 {CompileShiftExpr: just add expr} {expr 0xff-2} 253
test expr-10.3 {CompileShiftExpr: just add expr} {expr -1--2} 1
test expr-10.4 {CompileShiftExpr: just add expr} {expr 1-0123} -82
test expr-10.5 {CompileShiftExpr: error in add expr} {
    catch {expr x+3} msg
    set msg
} {syntax error in expression "x+3": variable references require preceding $}
test expr-10.6 {CompileShiftExpr: simple shift exprs} {expr 0xff>>0x3} 31
test expr-10.7 {CompileShiftExpr: simple shift exprs} {expr -0xf2<<0x3} -1936
test expr-10.8 {CompileShiftExpr: error compiling shift arm} {
    catch {expr 2**3>>6} msg
    set msg
} {syntax error in expression "2**3>>6": unexpected operator *}
test expr-10.9 {CompileShiftExpr: error compiling shift arm} {
    catch {expr 2<<x} msg
    set msg
} {syntax error in expression "2<<x": variable references require preceding $}
test expr-10.10 {CompileShiftExpr: runtime error} {
    list [catch {expr {24.0>>43}} msg] $msg
} {1 {can't use floating-point value as operand of ">>"}}
test expr-10.11 {CompileShiftExpr: runtime error} {
    list [catch {expr {"a"<<"b"}} msg] $msg
} {1 {can't use non-numeric string as operand of "<<"}}

test expr-11.1 {CompileAddExpr: just multiply expr} {expr 4*-2} -8
test expr-11.2 {CompileAddExpr: just multiply expr} {expr 0xff%2} 1
test expr-11.3 {CompileAddExpr: just multiply expr} {expr -1/2} -1
test expr-11.4 {CompileAddExpr: just multiply expr} {expr 7891%0123} 6
test expr-11.5 {CompileAddExpr: error in multiply expr} {
    catch {expr x*3} msg
    set msg
} {syntax error in expression "x*3": variable references require preceding $}
test expr-11.6 {CompileAddExpr: simple add exprs} {expr 0xff++0x3} 258
test expr-11.7 {CompileAddExpr: simple add exprs} {expr -0xf2--0x3} -239
test expr-11.8 {CompileAddExpr: error compiling add arm} {
    catch {expr 2**3+6} msg
    set msg
} {syntax error in expression "2**3+6": unexpected operator *}
test expr-11.9 {CompileAddExpr: error compiling add arm} {
    catch {expr 2-x} msg
    set msg
} {syntax error in expression "2-x": variable references require preceding $}
test expr-11.10 {CompileAddExpr: runtime error} {
    list [catch {expr {24.0+"xx"}} msg] $msg
} {1 {can't use non-numeric string as operand of "+"}}
test expr-11.11 {CompileAddExpr: runtime error} {
    list [catch {expr {"a"-"b"}} msg] $msg
} {1 {can't use non-numeric string as operand of "-"}}
test expr-11.12 {CompileAddExpr: runtime error} {
    list [catch {expr {3/0}} msg] $msg
} {1 {divide by zero}}
test expr-11.13 {CompileAddExpr: runtime error} {
    list [catch {expr {2.3/0.0}} msg] $msg
} {1 {divide by zero}}

test expr-12.1 {CompileMultiplyExpr: just unary expr} {expr ~4} -5
test expr-12.2 {CompileMultiplyExpr: just unary expr} {expr --5} 5
test expr-12.3 {CompileMultiplyExpr: just unary expr} {expr !27} 0
test expr-12.4 {CompileMultiplyExpr: just unary expr} {expr ~0xff00ff} -16711936
test expr-12.5 {CompileMultiplyExpr: error in unary expr} {
    catch {expr ~x} msg
    set msg
} {syntax error in expression "~x": variable references require preceding $}
test expr-12.6 {CompileMultiplyExpr: simple multiply exprs} {expr 0xff*0x3} 765
test expr-12.7 {CompileMultiplyExpr: simple multiply exprs} {expr -0xf2%-0x3} -2
test expr-12.8 {CompileMultiplyExpr: error compiling multiply arm} {
    catch {expr 2*3%%6} msg
    set msg
} {syntax error in expression "2*3%%6": unexpected operator %}
test expr-12.9 {CompileMultiplyExpr: error compiling multiply arm} {
    catch {expr 2*x} msg
    set msg
} {syntax error in expression "2*x": variable references require preceding $}
test expr-12.10 {CompileMultiplyExpr: runtime error} {
    list [catch {expr {24.0*"xx"}} msg] $msg
} {1 {can't use non-numeric string as operand of "*"}}
test expr-12.11 {CompileMultiplyExpr: runtime error} {
    list [catch {expr {"a"/"b"}} msg] $msg
} {1 {can't use non-numeric string as operand of "/"}}

test expr-13.1 {CompileUnaryExpr: unary exprs} {expr -0xff} -255
test expr-13.2 {CompileUnaryExpr: unary exprs} {expr +000123} 83
test expr-13.3 {CompileUnaryExpr: unary exprs} {expr +--++36} 36
test expr-13.4 {CompileUnaryExpr: unary exprs} {expr !2} 0
test expr-13.5 {CompileUnaryExpr: unary exprs} {expr +--+-62.0} -62.0
test expr-13.6 {CompileUnaryExpr: unary exprs} {expr !0.0} 1
test expr-13.7 {CompileUnaryExpr: unary exprs} {expr !0xef} 0
test expr-13.8 {CompileUnaryExpr: error compiling unary expr} {
    catch {expr ~x} msg
    set msg
} {syntax error in expression "~x": variable references require preceding $}
test expr-13.9 {CompileUnaryExpr: error compiling unary expr} {
    catch {expr !1.x} msg
    set msg
} {syntax error in expression "!1.x": extra tokens at end of expression}
test expr-13.10 {CompileUnaryExpr: runtime error} {
    list [catch {expr {~"xx"}} msg] $msg
} {1 {can't use non-numeric string as operand of "~"}}
test expr-13.11 {CompileUnaryExpr: runtime error} {
    list [catch {expr ~4.0} msg] $msg
} {1 {can't use floating-point value as operand of "~"}}
test expr-13.12 {CompileUnaryExpr: just primary expr} {expr 0x123} 291
test expr-13.13 {CompileUnaryExpr: just primary expr} {
    set a 27
    expr $a
} 27
test expr-13.14 {CompileUnaryExpr: just primary expr} {
    expr double(27)
} 27.0
test expr-13.15 {CompileUnaryExpr: just primary expr} {expr "123"} 123
test expr-13.16 {CompileUnaryExpr: error in primary expr} {
    catch {expr [set]} msg
    set msg
} {wrong # args: should be "set varName ?newValue?"}
test expr-13.17 {CompileUnaryExpr: negating non-numeric boolean literals} {
    set a1 yes; set a0 no; set b1 true; set b0 false
    list [expr {!$a1}] [expr {!$a0}] [expr {!$b1}] [expr {!$b0}]
} {0 1 0 1}

test expr-14.1 {CompilePrimaryExpr: literal primary} {expr 1} 1
test expr-14.2 {CompilePrimaryExpr: literal primary} {expr 123} 123
test expr-14.3 {CompilePrimaryExpr: literal primary} {expr 0xff} 255
test expr-14.4 {CompilePrimaryExpr: literal primary} {expr 00010} 8
test expr-14.5 {CompilePrimaryExpr: literal primary} {expr 62.0} 62.0
test expr-14.6 {CompilePrimaryExpr: literal primary} {
    expr 3.1400000
} 3.14
test expr-14.7 {CompilePrimaryExpr: literal primary} {expr {{abcde}<{abcdef}}} 1
test expr-14.8 {CompilePrimaryExpr: literal primary} {expr {{abc\
def} < {abcdef}}} 1
test expr-14.9 {CompilePrimaryExpr: literal primary} {expr {{abc\tde} > {abc\tdef}}} 0
test expr-14.10 {CompilePrimaryExpr: literal primary} {expr {{123}}} 123
test expr-14.11 {CompilePrimaryExpr: var reference primary} {
    set i 789
    list [expr {$i}] [expr $i]
} {789 789}
test expr-14.12 {CompilePrimaryExpr: var reference primary} {
    set i {789}    ;# test expr's aggressive conversion to numeric semantics
    list [expr {$i}] [expr $i]
} {789 789}
test expr-14.13 {CompilePrimaryExpr: var reference primary} {
    catch {unset a}
    set a(foo) foo
    set a(bar) bar
    set a(123) 123
    set result ""
    lappend result [expr $a(123)] [expr {$a(bar)<$a(foo)}]
    catch {unset a}
    set result
} {123 1}
test expr-14.14 {CompilePrimaryExpr: var reference primary} {
    set i 123    ;# test "$var.0" floating point conversion hack
    list [expr $i] [expr $i.0] [expr $i.0/12.0]
} {123 123.0 10.25}
test expr-14.15 {CompilePrimaryExpr: var reference primary} {
    set i 123
    catch {expr $i.2} msg
    set msg
} 123.2
test expr-14.16 {CompilePrimaryExpr: error compiling var reference primary} {
    catch {expr {$a(foo}} msg
    set errorInfo
} {missing )
    while compiling
"expr {$a(foo}"}
test expr-14.17 {CompilePrimaryExpr: string primary that looks like var ref} {
    expr $
} $
test expr-14.18 {CompilePrimaryExpr: quoted string primary} {
    expr "21"
} 21
test expr-14.19 {CompilePrimaryExpr: quoted string primary} {
    set i 123
    set x 456
    expr "$i+$x"
} 579
test expr-14.20 {CompilePrimaryExpr: quoted string primary} {
    set i 3
    set x 6
    expr 2+"$i.$x"
} 5.6
test expr-14.21 {CompilePrimaryExpr: error in quoted string primary} {
    catch {expr "[set]"} msg
    set msg
} {wrong # args: should be "set varName ?newValue?"}
test expr-14.22 {CompilePrimaryExpr: subcommand primary} {
    expr {[set i 123; set i]}
} 123
test expr-14.23 {CompilePrimaryExpr: error in subcommand primary} {
    catch {expr {[set]}} msg
    set errorInfo
} {wrong # args: should be "set varName ?newValue?"
    while compiling
"set"
    while compiling
"expr {[set]}"}
test expr-14.24 {CompilePrimaryExpr: error in subcommand primary} {
    catch {expr {[set i}} msg
    set errorInfo
} {missing close-bracket
    while compiling
"expr {[set i}"}
test expr-14.25 {CompilePrimaryExpr: math function primary} {
    format %.6g [expr exp(1.0)]
} 2.71828
test expr-14.26 {CompilePrimaryExpr: math function primary} {
    format %.6g [expr pow(2.0+0.1,3.0+0.1)]
} 9.97424
test expr-14.27 {CompilePrimaryExpr: error in math function primary} {
    catch {expr sinh::(2.0)} msg
    set errorInfo
} {syntax error in expression "sinh::(2.0)": expected parenthesis enclosing function arguments
    while compiling
"expr sinh::(2.0)"}
test expr-14.28 {CompilePrimaryExpr: subexpression primary} {
    expr 2+(3*4)
} 14
test expr-14.29 {CompilePrimaryExpr: error in subexpression primary} {
    catch {expr 2+(3*[set])} msg
    set errorInfo
} {wrong # args: should be "set varName ?newValue?"
    while compiling
"set"
    while compiling
"expr 2+(3*[set])"}
test expr-14.30 {CompilePrimaryExpr: missing paren in subexpression primary} {
    catch {expr 2+(3*(4+5)} msg
    set errorInfo
} {syntax error in expression "2+(3*(4+5)": looking for close parenthesis
    while compiling
"expr 2+(3*(4+5)"}
test expr-14.31 {CompilePrimaryExpr: just var ref in subexpression primary} {
    set i "5+10"
    list "[expr $i] == 15" "[expr ($i)] == 15" "[eval expr ($i)] == 15"
} {{15 == 15} {15 == 15} {15 == 15}}
test expr-14.32 {CompilePrimaryExpr: unexpected token} {
    catch {expr @} msg
    set errorInfo
} {syntax error in expression "@": character not legal in expressions
    while compiling
"expr @"}

test expr-15.1 {CompileMathFuncCall: missing parenthesis} {
    catch {expr sinh2.0)} msg
    set errorInfo
} {syntax error in expression "sinh2.0)": variable references require preceding $
    while compiling
"expr sinh2.0)"}
test expr-15.2 {CompileMathFuncCall: unknown math function} {
    catch {expr whazzathuh(1)} msg
    set errorInfo
} {unknown math function "whazzathuh"
    while compiling
"expr whazzathuh(1)"}
test expr-15.3 {CompileMathFuncCall: too many arguments} {
    catch {expr sin(1,2,3)} msg
    set errorInfo
} {too many arguments for math function
    while compiling
"expr sin(1,2,3)"}
test expr-15.4 {CompileMathFuncCall: ')' found before last required arg} {
    catch {expr sin()} msg
    set errorInfo
} {too few arguments for math function
    while compiling
"expr sin()"}
test expr-15.5 {CompileMathFuncCall: too few arguments} {
    catch {expr pow(1)} msg
    set errorInfo
} {too few arguments for math function
    while compiling
"expr pow(1)"}
test expr-15.6 {CompileMathFuncCall: missing ')'} {
    catch {expr sin(1} msg
    set errorInfo
} {syntax error in expression "sin(1": missing close parenthesis at end of function call
    while compiling
"expr sin(1"}
test expr-15.7 {CompileMathFuncCall: call registered math function} {registeredMathFuncs} {
    expr 2*T1()
} 246
test expr-15.8 {CompileMathFuncCall: call registered math function} {registeredMathFuncs} {
    expr T2()*3
} 1035
test expr-15.9 {CompileMathFuncCall: call registered math function} {registeredMathFuncs} {
    expr T3(21, 37)
} 37
test expr-15.10 {CompileMathFuncCall: call registered math function} {registeredMathFuncs} {
    expr T3(21.2, 37)
} 37.0
test expr-15.11 {CompileMathFuncCall: call registered math function} {registeredMathFuncs} {
    expr T3(-21.2, -17.5)
} -17.5
test expr-15.12 {ExprCallMathFunc: call registered math function} {registeredMathFuncs} {
    expr T3(21, wide(37))
} 37
test expr=15.13 {ExprCallMathFunc: call registered math function} {registeredMathFuncs} {
    expr T3(wide(21), 37)
} 37
test expr=15.14 {ExprCallMathFunc: call registered math function} {registeredMathFuncs} {
    expr T3(wide(21), wide(37))
} 37
test expr-15.15 {ExprCallMathFunc: call registered math function} {registeredMathFuncs} {
    expr T3(21.0, wide(37))
} 37.0
test expr=15.16 {ExprCallMathFunc: call registered math function} {registeredMathFuncs} {
    expr T3(wide(21), 37.0)
} 37.0

test expr-16.1 {GetToken: checks whether integer token starting with "0x" (e.g., "0x$") is invalid} {
    catch {unset a}
    set a(VALUE) ff15
    set i 123
    if {[expr 0x$a(VALUE)] & 16} {
        set i {}
    }
    set i
} {}
test expr-16.2 {GetToken: check for string literal in braces} {
    expr {{1}}
} {1}

# Check "expr" and computed command names.

test expr-17.1 {expr and computed command names} {
    set i 0
    set z expr
    $z 1+2
} 3

# Check correct conversion of operands to numbers: If the string looks like
# an integer, convert to integer. Otherwise, if the string looks like a
# double, convert to double.

test expr-18.1 {expr and conversion of operands to numbers} {
    set x [lindex 11 0]
    catch {expr int($x)}
    expr {$x}
} 11
test expr-18.2 {whitespace strings should not be == 0 (buggy strtod)} {
    expr {" "}
} { }

# Check "expr" and interpreter result object resetting before appending
# an error msg during evaluation of exprs not in {}s

test expr-19.1 {expr and interpreter result object resetting} {
    proc p {} {
        set t  10.0
        set x  2.0
        set dx 0.2
        set f  {$dx-$x/10}
        set g  {-$x/5}
        set center 1.0
        set x  [expr $x-$center]
        set dx [expr $dx+$g]
        set x  [expr $x+$f+$center]
        set x  [expr $x+$f+$center]
        set y  [expr round($x)]
    }
    p
} 3

# Test for incorrect "double evaluation" semantics

test expr-20.1 {wrong brace matching} {
    catch {unset l}
    catch {unset r}
    catch {unset q}
    catch {unset cmd}
    catch {unset a}
    set l "\{"; set r "\}"; set q "\""
    set cmd "expr $l$q|$q == $q$r$q$r"
    list [catch $cmd a] $a
} {1 {extra characters after close-brace}}
test expr-20.2 {double invocation of variable traces} {
    set exprtracecounter 0
    proc exprtraceproc {args} {
       upvar #0 exprtracecounter counter
       set argc [llength $args]
       set extraargs [lrange $args 0 [expr {$argc - 4}]]
       set name [lindex $args [expr {$argc - 3}]]
       upvar 1 $name var
       if {[incr counter] % 2 == 1} {
           set var "$counter oops [concat $extraargs]"
       } else {
           set var "$counter + [concat $extraargs]"
       }
    }
    trace variable exprtracevar r [list exprtraceproc 10]
    list [catch {expr "$exprtracevar + 20"} a] $a \
        [catch {expr "$exprtracevar + 20"} b] $b \
        [unset exprtracevar exprtracecounter]
} {1 {syntax error in expression "1 oops 10 + 20": extra tokens at end of expression} 0 32 {}}
test expr-20.3 {broken substitution of integer digits} {
    # fails with 8.0.x, but not 8.1b2
    list [set a 000; expr 0x1$a] [set a 1; expr ${a}000]
} {4096 1000}
test expr-20.4 {proper double evaluation compilation, error case} {
    catch {unset a}; # make sure $a doesn't exist
    list [catch {expr 1?{$a}:0} msg] $msg
} {1 {can't read "a": no such variable}}
test expr-20.5 {proper double evaluation compilation, working case} {
    set a yellow
    expr 1?{$a}:0
} yellow
test expr-20.6 {handling of compile error in trial compile} {
    list [catch {expr + {[incr]}} msg] $msg
} {1 {wrong # args: should be "incr varName ?increment?"}}
test expr-20.7 {handling of compile error in runtime case} {
    list [catch {expr + {[error foo]}} msg] $msg
} {1 foo}

# Test for non-numeric boolean literal handling
test expr-21.1 	{non-numeric boolean literals} {expr false } false
test expr-21.2 	{non-numeric boolean literals} {expr true  } true
test expr-21.3 	{non-numeric boolean literals} {expr off   } off
test expr-21.4 	{non-numeric boolean literals} {expr on    } on
test expr-21.5 	{non-numeric boolean literals} {expr no    } no
test expr-21.6 	{non-numeric boolean literals} {expr yes   } yes
test expr-21.7 	{non-numeric boolean literals} {expr !false} 1
test expr-21.8 	{non-numeric boolean literals} {expr !true } 0
test expr-21.9 	{non-numeric boolean literals} {expr !off  } 1
test expr-21.10 {non-numeric boolean literals} {expr !on   } 0
test expr-21.11 {non-numeric boolean literals} {expr !no   } 1
test expr-21.12 {non-numeric boolean literals} {expr !yes  } 0

# Test for non-numeric float handling.
#
# These are non-portable because strtod()-support for "Inf" and "NaN"
# is so wildly variable.  This sucks...
test expr-22.1 {non-numeric floats} nonPortable {
    list [catch {expr {NaN + 1}} msg] $msg
} {1 {can't use non-numeric floating-point value as operand of "+"}}
test expr-22.2 {non-numeric floats} nonPortable {
    list [catch {expr {Inf + 1}} msg] $msg
} {1 {can't use infinite floating-point value as operand of "+"}}
test expr-22.3 {non-numeric floats} nonPortable {
    set nan NaN
    list [catch {expr {$nan + 1}} msg] $msg
} {1 {can't use non-numeric floating-point value as operand of "+"}}
test expr-22.4 {non-numeric floats} nonPortable {
    set inf Inf
    list [catch {expr {$inf + 1}} msg] $msg
} {1 {can't use infinite floating-point value as operand of "+"}}
test expr-22.5 {non-numeric floats} nonPortable {
    list [catch {expr NaN} msg] $msg
} {1 {domain error: argument not in valid range}}
test expr-22.6 {non-numeric floats} nonPortable {
    list [catch {expr Inf} msg] $msg
} {1 {floating-point value too large to represent}}
test expr-22.7 {non-numeric floats} nonPortable {
    list [catch {expr {1 / NaN}} msg] $msg
} {1 {can't use non-numeric floating-point value as operand of "/"}}
test expr-22.8 {non-numeric floats} nonPortable {
    list [catch {expr {1 / Inf}} msg] $msg
} {1 {can't use infinite floating-point value as operand of "/"}}
# Make sure [Bug 761471] stays fixed.
test expr-22.9 {non-numeric floats: shared object equality and NaN} {
    set x NaN
    expr {$x == $x}
} 0

# Some compilers get this wrong; ensure that we work around it correctly
test expr-24.1 {expr edge cases; shifting} {expr int(5)>>31} 0
test expr-24.2 {expr edge cases; shifting} {expr int(5)>>63} 0
test expr-24.3 {expr edge cases; shifting} {expr wide(5)>>31} 0
test expr-24.4 {expr edge cases; shifting} {expr wide(5)>>63} 0
test expr-24.5 {expr edge cases; shifting} nonPortable {expr int(5)<<31} 0
test expr-24.6 {expr edge cases; shifting} nonPortable {expr int(5)<<63} 0
test expr-24.7 {expr edge cases; shifting} {expr wide(5)<<31} 10737418240
test expr-24.8 {expr edge cases; shifting} nonPortable {expr wide(5)<<63} -9223372036854775808
test expr-24.9 {expr edge cases; shifting} {expr 5>>32} 0

# cleanup
if {[info exists a]} {
    unset a
}
::tcltest::cleanupTests
return
