/**
 * Copyright (c) 2022 DB Netz AG and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v20.html
 */
package org.eclipse.set.toolboxmodel.transform.internal;

import com.google.common.collect.Iterables;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.set.toolboxmodel.Basisobjekte.Identitaet_TypeClass;
import org.eclipse.set.toolboxmodel.Basisobjekte.Ur_Objekt;
import org.eclipse.set.toolboxmodel.PlanPro.Ausgabe_Fachdaten;
import org.eclipse.set.toolboxmodel.PlanPro.Container_AttributeGroup;
import org.eclipse.set.toolboxmodel.PlanPro.Fachdaten_AttributeGroup;
import org.eclipse.set.toolboxmodel.PlanPro.LST_Planung_AttributeGroup;
import org.eclipse.set.toolboxmodel.PlanPro.LST_Zustand;
import org.eclipse.set.toolboxmodel.PlanPro.PlanPro_Schnittstelle;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Guid to Object cache for faster lookups
 * 
 * @author Stuecker
 */
@SuppressWarnings("all")
public class GuidCache {
  public enum ContainerType {
    Single,

    Initial,

    Planning,

    Global;
  }

  private final Map<String, EObject> globalGuidToObjectMap = new HashMap<String, EObject>();

  private final Map<String, EObject> singleGuidToObjectMap = new HashMap<String, EObject>();

  private final Map<String, EObject> initialGuidToObjectMap = new HashMap<String, EObject>();

  private final Map<String, EObject> planningGuidToObjectMap = new HashMap<String, EObject>();

  /**
   * @param guid the GUID
   * @param source the source object which references the guid
   * @return the referenced object or null if it cannot be found
   */
  public EObject get(final String guid, final EObject source) {
    final GuidCache.ContainerType container = this.getContainerType(source);
    if (container != null) {
      switch (container) {
        case Initial:
          return this.initialGuidToObjectMap.get(guid);
        case Planning:
          return this.planningGuidToObjectMap.get(guid);
        case Single:
          return this.singleGuidToObjectMap.get(guid);
        case Global:
          return this.globalGuidToObjectMap.get(guid);
        default:
          return null;
      }
    } else {
      return null;
    }
  }

  /**
   * Initialize the cache
   * 
   * @param planProSchnittstelle the PlanPro_Schnittstelle
   */
  public void prepare(final PlanPro_Schnittstelle planProSchnittstelle) {
    this.prepare(planProSchnittstelle, GuidCache.ContainerType.Global, this.globalGuidToObjectMap);
    this.prepare(planProSchnittstelle, GuidCache.ContainerType.Single, this.singleGuidToObjectMap);
    this.prepare(planProSchnittstelle, GuidCache.ContainerType.Initial, this.initialGuidToObjectMap);
    this.prepare(planProSchnittstelle, GuidCache.ContainerType.Planning, this.planningGuidToObjectMap);
  }

  private void prepare(final PlanPro_Schnittstelle planProSchnittstelle, final GuidCache.ContainerType containerType, final Map<String, EObject> map) {
    Iterable<Ur_Objekt> _commonObjects = this.getCommonObjects(planProSchnittstelle);
    Iterable<Ur_Objekt> _elvis = null;
    Iterable<Ur_Objekt> _objectList = this.getObjectList(planProSchnittstelle, containerType);
    if (_objectList != null) {
      _elvis = _objectList;
    } else {
      _elvis = Collections.<Ur_Objekt>unmodifiableList(CollectionLiterals.<Ur_Objekt>newArrayList());
    }
    final Consumer<Ur_Objekt> _function = new Consumer<Ur_Objekt>() {
      @Override
      public void accept(final Ur_Objekt it) {
        Identitaet_TypeClass _identitaet = null;
        if (it!=null) {
          _identitaet=it.getIdentitaet();
        }
        String _wert = null;
        if (_identitaet!=null) {
          _wert=_identitaet.getWert();
        }
        map.put(_wert, it);
      }
    };
    Iterables.<Ur_Objekt>concat(_commonObjects, _elvis).forEach(_function);
  }

  private Iterable<Ur_Objekt> getObjectList(final PlanPro_Schnittstelle planProSchnittstelle, final GuidCache.ContainerType containerType) {
    if (containerType != null) {
      switch (containerType) {
        case Single:
          LST_Zustand _lSTZustand = null;
          if (planProSchnittstelle!=null) {
            _lSTZustand=planProSchnittstelle.getLSTZustand();
          }
          Container_AttributeGroup _container = null;
          if (_lSTZustand!=null) {
            _container=_lSTZustand.getContainer();
          }
          EList<EObject> _eContents = null;
          if (_container!=null) {
            _eContents=_container.eContents();
          }
          Iterable<Ur_Objekt> _filter = null;
          if (_eContents!=null) {
            _filter=Iterables.<Ur_Objekt>filter(_eContents, Ur_Objekt.class);
          }
          return _filter;
        case Planning:
          LST_Planung_AttributeGroup _lSTPlanung = null;
          if (planProSchnittstelle!=null) {
            _lSTPlanung=planProSchnittstelle.getLSTPlanung();
          }
          Fachdaten_AttributeGroup _fachdaten = null;
          if (_lSTPlanung!=null) {
            _fachdaten=_lSTPlanung.getFachdaten();
          }
          EList<Ausgabe_Fachdaten> _ausgabeFachdaten = null;
          if (_fachdaten!=null) {
            _ausgabeFachdaten=_fachdaten.getAusgabeFachdaten();
          }
          Iterable<EObject> _flatMap = null;
          if (_ausgabeFachdaten!=null) {
            final Function1<Ausgabe_Fachdaten, EList<EObject>> _function = new Function1<Ausgabe_Fachdaten, EList<EObject>>() {
              @Override
              public EList<EObject> apply(final Ausgabe_Fachdaten it) {
                LST_Zustand _lSTZustandZiel = it.getLSTZustandZiel();
                Container_AttributeGroup _container = null;
                if (_lSTZustandZiel!=null) {
                  _container=_lSTZustandZiel.getContainer();
                }
                EList<EObject> _eContents = null;
                if (_container!=null) {
                  _eContents=_container.eContents();
                }
                return _eContents;
              }
            };
            _flatMap=IterableExtensions.<Ausgabe_Fachdaten, EObject>flatMap(_ausgabeFachdaten, _function);
          }
          Iterable<Ur_Objekt> _filter_1 = null;
          if (_flatMap!=null) {
            _filter_1=Iterables.<Ur_Objekt>filter(_flatMap, Ur_Objekt.class);
          }
          return _filter_1;
        case Initial:
          LST_Planung_AttributeGroup _lSTPlanung_1 = null;
          if (planProSchnittstelle!=null) {
            _lSTPlanung_1=planProSchnittstelle.getLSTPlanung();
          }
          Fachdaten_AttributeGroup _fachdaten_1 = null;
          if (_lSTPlanung_1!=null) {
            _fachdaten_1=_lSTPlanung_1.getFachdaten();
          }
          EList<Ausgabe_Fachdaten> _ausgabeFachdaten_1 = null;
          if (_fachdaten_1!=null) {
            _ausgabeFachdaten_1=_fachdaten_1.getAusgabeFachdaten();
          }
          Iterable<EObject> _flatMap_1 = null;
          if (_ausgabeFachdaten_1!=null) {
            final Function1<Ausgabe_Fachdaten, EList<EObject>> _function_1 = new Function1<Ausgabe_Fachdaten, EList<EObject>>() {
              @Override
              public EList<EObject> apply(final Ausgabe_Fachdaten it) {
                LST_Zustand _lSTZustandStart = it.getLSTZustandStart();
                Container_AttributeGroup _container = null;
                if (_lSTZustandStart!=null) {
                  _container=_lSTZustandStart.getContainer();
                }
                EList<EObject> _eContents = null;
                if (_container!=null) {
                  _eContents=_container.eContents();
                }
                return _eContents;
              }
            };
            _flatMap_1=IterableExtensions.<Ausgabe_Fachdaten, EObject>flatMap(_ausgabeFachdaten_1, _function_1);
          }
          Iterable<Ur_Objekt> _filter_2 = null;
          if (_flatMap_1!=null) {
            _filter_2=Iterables.<Ur_Objekt>filter(_flatMap_1, Ur_Objekt.class);
          }
          return _filter_2;
        case Global:
          LST_Planung_AttributeGroup _lSTPlanung_2 = null;
          if (planProSchnittstelle!=null) {
            _lSTPlanung_2=planProSchnittstelle.getLSTPlanung();
          }
          Fachdaten_AttributeGroup _fachdaten_2 = null;
          if (_lSTPlanung_2!=null) {
            _fachdaten_2=_lSTPlanung_2.getFachdaten();
          }
          EList<Ausgabe_Fachdaten> _ausgabeFachdaten_2 = null;
          if (_fachdaten_2!=null) {
            _ausgabeFachdaten_2=_fachdaten_2.getAusgabeFachdaten();
          }
          Iterable<EObject> _flatMap_2 = null;
          if (_ausgabeFachdaten_2!=null) {
            final Function1<Ausgabe_Fachdaten, Iterable<EObject>> _function_2 = new Function1<Ausgabe_Fachdaten, Iterable<EObject>>() {
              @Override
              public Iterable<EObject> apply(final Ausgabe_Fachdaten it) {
                LST_Zustand _lSTZustandStart = it.getLSTZustandStart();
                LST_Zustand _lSTZustandZiel = it.getLSTZustandZiel();
                final Function1<LST_Zustand, EList<EObject>> _function = new Function1<LST_Zustand, EList<EObject>>() {
                  @Override
                  public EList<EObject> apply(final LST_Zustand it_1) {
                    Container_AttributeGroup _container = it_1.getContainer();
                    EList<EObject> _eContents = null;
                    if (_container!=null) {
                      _eContents=_container.eContents();
                    }
                    return _eContents;
                  }
                };
                return IterableExtensions.<LST_Zustand, EObject>flatMap(Collections.<LST_Zustand>unmodifiableList(CollectionLiterals.<LST_Zustand>newArrayList(_lSTZustandStart, _lSTZustandZiel)), _function);
              }
            };
            _flatMap_2=IterableExtensions.<Ausgabe_Fachdaten, EObject>flatMap(_ausgabeFachdaten_2, _function_2);
          }
          Iterable<Ur_Objekt> _filter_3 = null;
          if (_flatMap_2!=null) {
            _filter_3=Iterables.<Ur_Objekt>filter(_flatMap_2, Ur_Objekt.class);
          }
          return _filter_3;
        default:
          break;
      }
    }
    return null;
  }

  private Iterable<Ur_Objekt> getCommonObjects(final EObject object) {
    if ((object instanceof Container_AttributeGroup)) {
      return Collections.<Ur_Objekt>unmodifiableList(CollectionLiterals.<Ur_Objekt>newArrayList());
    }
    if ((!(object instanceof Ur_Objekt))) {
      final Function1<EObject, Iterable<Ur_Objekt>> _function = new Function1<EObject, Iterable<Ur_Objekt>>() {
        @Override
        public Iterable<Ur_Objekt> apply(final EObject it) {
          return GuidCache.this.getCommonObjects(it);
        }
      };
      return IterableExtensions.<EObject, Ur_Objekt>flatMap(object.eContents(), _function);
    }
    final Function1<EObject, Iterable<Ur_Objekt>> _function_1 = new Function1<EObject, Iterable<Ur_Objekt>>() {
      @Override
      public Iterable<Ur_Objekt> apply(final EObject it) {
        return GuidCache.this.getCommonObjects(it);
      }
    };
    Iterable<Ur_Objekt> _flatMap = IterableExtensions.<EObject, Ur_Objekt>flatMap(object.eContents(), _function_1);
    return Iterables.<Ur_Objekt>concat(Collections.<Ur_Objekt>unmodifiableList(CollectionLiterals.<Ur_Objekt>newArrayList(((Ur_Objekt) object))), _flatMap);
  }

  private GuidCache.ContainerType getContainerType(final EObject referenceObject) {
    if ((referenceObject == null)) {
      return GuidCache.ContainerType.Global;
    }
    if ((!(referenceObject instanceof Container_AttributeGroup))) {
      return this.getContainerType(referenceObject.eContainer());
    }
    EObject lstState = referenceObject.eContainer();
    EObject lstStateContainer = lstState.eContainer();
    if ((!(lstStateContainer instanceof Ausgabe_Fachdaten))) {
      return GuidCache.ContainerType.Single;
    }
    Ausgabe_Fachdaten ausgabeFachdaten = ((Ausgabe_Fachdaten) lstStateContainer);
    LST_Zustand _lSTZustandStart = ausgabeFachdaten.getLSTZustandStart();
    boolean _tripleEquals = (_lSTZustandStart == lstState);
    if (_tripleEquals) {
      return GuidCache.ContainerType.Initial;
    } else {
      return GuidCache.ContainerType.Planning;
    }
  }
}
