/*******************************************************************************
 * Copyright (c) 1998, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation from Oracle TopLink
******************************************************************************/
package org.eclipse.persistence.tools.db.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import org.eclipse.persistence.tools.utility.collection.CollectionTools;
import org.eclipse.persistence.tools.utility.node.AbstractNode;
import org.eclipse.persistence.tools.utility.node.Node;

/**
 * @version 2.6
 */
public abstract class ELModel extends AbstractNode
                              implements ELNode {

	// ********** constructors **********

	/**
	 * Every object (except MWProject) must have a parent.
	 * Use this constructor to create a new object - the default
	 * constructor should only be used by TopLink.
	 * @see #initialize(Node)
	 */
	protected ELModel(ELNode parent) {
		super(parent);
	}


	// ********** convenience methods **********

	/**
	 * Returns the object's parent in the containment hierarchy.
	 * Every object except the MWProject will have a parent.
	 */
	// TODO this can be renamed to getParent() when we move to jdk 1.5
	@Override
	public final ELNode getMWParent() {
		return (ELNode) getParent();
	}

	// ********** model synchronization support **********

	/**
	 * Convenience method.
	 * Remove the specified node from the specified bound collection,
	 * firing the appropriate change event if necessary and
	 * launching the cascading #nodeRemoved() mechanism.
	 * Returns true if the collection changed - @see java.util.Collection.remove(Object).
	 */
	protected boolean removeNodeFromCollection(Node node, Collection collection, String collectionName) {
		boolean changed = this.removeItemFromCollection(node, collection, collectionName);
		return changed;
	}

	/**
	 * Convenience method.
	 * Remove the specified node from the specified collection
	 * and launch the cascading #nodeRemoved() mechanism.
	 * No change event is fired.
	 * Returns true if the collection changed - @see java.util.Collection.remove(Object).
	 */
	protected boolean removeNodeFromCollection(Node node, Collection collection) {
		boolean changed = collection.remove(node);
		return changed;
	}

	/**
	 * Convenience method.
	 * Remove the specified nodes from the specified bound collection,
	 * firing the appropriate change event if necessary and
	 * launching the cascading #nodeRemoved() mechanism.
	 * Returns true if the collection changed - @see java.util.Collection.remove(Object).
	 */
	protected boolean removeNodesFromCollection(Collection nodes, Collection collection, String collectionName) {
		return this.removeNodesFromCollection(nodes.iterator(), collection, collectionName);
	}

	/**
	 * Convenience method.
	 * Remove the specified nodes from the specified bound collection,
	 * firing the appropriate change event if necessary,
	 * and launch the cascading #nodeRemoved() mechanism.
	 * Returns true if the collection changed - @see java.util.Collection.remove(Object).
	 */
	protected boolean removeNodesFromCollection(Iterator nodes, Collection collection, String collectionName) {
		Collection removedNodes = CollectionTools.collection(nodes);
		removedNodes.retainAll(collection);
		boolean changed = this.removeItemsFromCollection(nodes, collection, collectionName);
		return changed;
	}

	/**
	 * Convenience method.
	 * Remove the specified nodes from the specified collection
	 * and launch the cascading #nodeRemoved() mechanism.
	 * No change event is fired.
	 * Returns true if the collection changed - @see java.util.Collection.remove(Object).
	 */
	protected boolean removeNodesFromCollection(Collection nodes, Collection collection) {
		return this.removeNodesFromCollection(nodes.iterator(), collection);
	}

	/**
	 * Convenience method.
	 * Remove the specified nodes from the specified collection
	 * and launch the cascading #nodeRemoved() mechanism.
	 * No change event is fired.
	 * Returns true if the collection changed - @see java.util.Collection.remove(Object).
	 */
	protected boolean removeNodesFromCollection(Iterator nodes, Collection collection) {
		Collection removedNodes = CollectionTools.collection(nodes);
		removedNodes.retainAll(collection);
		boolean changed = collection.removeAll(removedNodes);
		return changed;
	}

	/**
	 * Convenience method.
	 * Clear the specified collection, firing the appropriate change event
	 * if necessary and launching the cascading #nodeRemoved() mechanism.
	 */
	protected void clearNodeCollection(Collection nodeCollection, String collectionName) {
		Collection removedNodes = new ArrayList(nodeCollection);
		this.clearCollection(nodeCollection, collectionName);
	}

	/**
	 * Convenience method.
	 * Clear the specified collection and launch the cascading #nodeRemoved()
	 * mechanism.
	 * No change event is fired.
	 */
	protected void clearNodeCollection(Collection nodeCollection) {
		Collection removedNodes = new ArrayList(nodeCollection);
		nodeCollection.clear();
	}

	/**
	 * Convenience method.
	 * Remove the node specified by index from the specified bound list,
	 * firing the appropriate change event if necessary and
	 * launching the cascading #nodeRemoved() mechanism.
	 * Returns the removed object - @see java.util.List.remove(int).
	 */
	protected Object removeNodeFromList(int index, List nodeList, String listName) {
		Node removedNode = (Node) this.removeItemFromList(index, nodeList, listName);
		return removedNode;
	}

	/**
	 * Convenience method.
	 * Remove the node specified by index from the specified list
	 * and launch the cascading #nodeRemoved() mechanism.
	 * No change event is fired.
	 * Returns the removed object - @see java.util.List.remove(int).
	 */
	protected Object removeNodeFromList(int index, List list) {
		Node removedNode = (Node) list.remove(index);
		return removedNode;
	}

	/**
	 * Convenience method.
	 * Clear the specified list, firing the appropriate change event if necessary
	 * and launching the cascading #nodeRemoved() mechanism.
	 */
	protected void clearNodeList(List nodeList, String listName) {
		Collection removedNodes = new ArrayList(nodeList);
		this.clearList(nodeList, listName);
	}

	/**
	 * Convenience method.
	 * Clear the specified list and launch the cascading #nodeRemoved() mechanism.
	 * No change event is fired.
	 */
	protected void clearNodeList(List nodeList) {
		// since no events thrown, same as for collection
		this.clearNodeCollection(nodeList);
	}

	// ********** display methods **********

	/**
	 * Returns something useful for the UI.
	 */
	@Override
	public String displayString() {
		// the default is to use the developer-friendly string...
		return this.toString();
	}
}