/*******************************************************************************
 * Copyright (c) 1998, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation from Oracle TopLink
******************************************************************************/
package org.eclipse.persistence.tools.db.model;

import org.eclipse.persistence.tools.db.model.handles.MWHandle;
import org.eclipse.persistence.tools.utility.node.Node;

/**
 * <code>ELLoginSpecHandle</code> is used to isolate the painful bits of code necessary to correctly
 * handle references to {@link ELLoginSpec} objects. Since a {@link ELLoginSpec} is nested within
 * the XML file for a MWDatabase, we need to store a reference to a particular login spec as
 *   - the name of the login spec
 * <p>
 * This causes no end of pain when dealing with TopLink, property change listeners, backward-
 * compatibility, etc.
 * <p>
 * NB: Currently this class is only used by MWDatabase. The model synchronization is performed
 * locally by that class since it holds the child collection of login specs ('loginSpecs') as well
 * as thereferences to them ('deploymentLoginSpec' and 'developmentLoginSpec'). If this handle is
 * ever used in another place in the model, the model synchronization will need to cascade to all
 * model objects
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public final class ELLoginSpecHandle extends MWHandle {

	/**
	 * This is the actual login spec.
	 * It is built from the login spec name, below.
	 */
	private volatile ELLoginSpec loginSpec;

	/**
	 * The login spec name is transient. It is used only to hold its value
	 * until #postProjectBuild() is called and we can resolve
	 * the actual login spec. We do not keep it in synch with the login spec
	 * itself because we cannot know when the login spec has been renamed etc.
	 */
	private volatile String loginSpecName;

	// ********** constructors **********

	/**
	 * default constructor - for TopLink use only
	 */
	@SuppressWarnings("unused")
	private ELLoginSpecHandle() {
		super();
	}

	ELLoginSpecHandle(ELDatabase parent, NodeReferenceScrubber scrubber) {
		super(parent, scrubber);
	}

	ELLoginSpecHandle(ELDatabase parent, ELLoginSpec loginSpec, NodeReferenceScrubber scrubber) {
		super(parent, scrubber);
		this.loginSpec = loginSpec;
	}


	// ********** instance methods **********

	ELLoginSpec getLoginSpec() {
		return this.loginSpec;
	}

	void setLoginSpec(ELLoginSpec loginSpec) {
		this.loginSpec = loginSpec;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Node node() {
		return getLoginSpec();
	}

	public ELLoginSpecHandle setScrubber(NodeReferenceScrubber scrubber) {
		this.setScrubberInternal(scrubber);
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void toString(StringBuffer sb) {
		if (this.loginSpec == null) {
			sb.append("null");
		} else {
			this.loginSpec.toString(sb);
		}
	}
}