/*******************************************************************************
 * Copyright (c) 2009, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 ******************************************************************************/
package org.eclipse.persistence.tools.db.driver;

import java.util.HashMap;
import org.eclipse.persistence.tools.gen.db.Database;

/**
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public class WorkbenchDriverAdapterManager {

	private final HashMap<String, WorkbenchDriverAdapterFactory> adapterFactories;
	private static final WorkbenchDriverAdapterFactory UNRECOGNIZED_ADAPTER_FACTORY = new Unknown.Factory();

	// singleton
	private static final WorkbenchDriverAdapterManager INSTANCE = new WorkbenchDriverAdapterManager();

	/**
	 * Returns the singleton.
	 */
	public static WorkbenchDriverAdapterManager instance() {
		return INSTANCE;
	}

	/**
	 * Ensure single instance.
	 */
	private WorkbenchDriverAdapterManager() {
		super();
		this.adapterFactories = this.buildAdapterFactories();
	}

	private HashMap<String, WorkbenchDriverAdapterFactory> buildAdapterFactories() {
		HashMap<String, WorkbenchDriverAdapterFactory> factories = new HashMap<String, WorkbenchDriverAdapterFactory>();
		this.addAdapterFactoriesTo(factories);
		return factories;
	}

	private void addAdapterFactoriesTo(HashMap<String, WorkbenchDriverAdapterFactory> factories) {
		this.addAdapterFactoryTo(new DB2.Factory(),        factories);
		this.addAdapterFactoryTo(new Derby.Factory(),      factories);
		this.addAdapterFactoryTo(new HSQLDB.Factory(),     factories);
		this.addAdapterFactoryTo(new Informix.Factory(),   factories);
		this.addAdapterFactoryTo(new MaxDB.Factory(),      factories);
		this.addAdapterFactoryTo(new MySQL.Factory(),      factories);
		this.addAdapterFactoryTo(new Oracle.Factory(),     factories);
		this.addAdapterFactoryTo(new PostgreSQL.Factory(), factories);
		this.addAdapterFactoryTo(new SQLServer.Factory(),  factories);
		this.addAdapterFactoryTo(new Sybase.Factory(),     factories);
	}

	/**
	 * @see org.eclipse.datatools.modelbase.sql.schema.Database#getVendor()
	 */
	private void addAdapterFactoryTo(WorkbenchDriverAdapterFactory factory, HashMap<String, WorkbenchDriverAdapterFactory> factories) {
		for (String name : factory.getSupportedVendors()) {
			this.addAdapterFactoryTo(name, factory, factories);
		}
	}

	private void addAdapterFactoryTo(String name, WorkbenchDriverAdapterFactory factory, HashMap<String, WorkbenchDriverAdapterFactory> factories) {
		if (factories.put(name, factory) != null) {
			throw new IllegalArgumentException("Duplicate adapter factory: " + name);
		}
	}

	public WorkbenchDriverAdapter buildAdapter(String dtpVendorName, Database database) {
		return this.getAdapterFactory(dtpVendorName).buildAdapter(database);
	}

	private WorkbenchDriverAdapterFactory getAdapterFactory(String dtpVendorName) {
		WorkbenchDriverAdapterFactory factory = this.adapterFactories.get(dtpVendorName);
		return (factory != null) ? factory : UNRECOGNIZED_ADAPTER_FACTORY;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return this.getClass().getSimpleName();
	}
}