/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.FetchType;
import org.eclipse.persistence.tools.mapping.orm.ExternalMappingVisitor;
import org.eclipse.persistence.tools.mapping.orm.ExternalReadTransformer;
import org.eclipse.persistence.tools.mapping.orm.ExternalTransformationMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalTransformer;
import org.eclipse.persistence.tools.mapping.orm.ExternalWriteTransformer;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form of a transformation mapping.
 *
 * @see EmbeddableEntity
 *
 * @version 2.6
 */
final class TransformationMapping extends NonTransientMapping
                                  implements ExternalTransformationMapping {

	/**
	 * Creates a new <code>TransformationMapping</code>.
	 *
	 * @param parent The parent of this external form
	 */
	TransformationMapping(Embeddable parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void accept(ExternalMappingVisitor visitor) {
		visitor.visit(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTransformer addReadTransformer() {
		ReadTransformer transformer = buildReadTransformer();
		transformer.addSelf();
		return transformer;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalWriteTransformer addWriteTransformer() {
		WriteTransfomer writeTransformer = buildWriteTransformer(-1);
		writeTransformer.addSelf();
		return writeTransformer;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(FETCH);
		names.add(OPTIONAL);
		names.add(ACCESS);
		names.add(MUTABLE);
		names.add(ATTRIBUTE_TYPE);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(ReadTransformer.READ_TRANSFORMER);
		names.add(WriteTransfomer.WRITE_TRANSFORMER);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private ReadTransformer buildReadTransformer() {
		return new ReadTransformer(this);
	}

	private WriteTransfomer buildWriteTransformer(int index) {
		return new WriteTransfomer(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return TRANSFORMATION;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FetchType getFetchType() {
		return getEnumAttribute(FETCH, FetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getFetchTypeTextRange() {
		return getAttributeTextRange(FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMutableTextRange() {
		return getAttributeTextRange(MUTABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getOptionalTextRange() {
		return getAttributeTextRange(OPTIONAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalReadTransformer getReadTransformer() {

		if (hasChild(ReadTransformer.READ_TRANSFORMER)) {
			return buildReadTransformer();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalWriteTransformer getWriteTransformer(int index) {

		if (hasChild(WriteTransfomer.WRITE_TRANSFORMER, index)) {
			return buildWriteTransformer(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isMutable() {
		return getBooleanAttribute(MUTABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isOptional() {
		return getBooleanAttribute(OPTIONAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeReadTransformer() {
		ReadTransformer transformer = buildReadTransformer();
		transformer.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeWriteTransformer(int index) {
		WriteTransfomer writeTransformer = buildWriteTransformer(index);
		writeTransformer.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setFetchType(FetchType type) {
		setAttribute(FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setIsMutable(Boolean mutable) {
		setAttribute(MUTABLE, mutable);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOptional(Boolean optional) {
		setAttribute(OPTIONAL, optional);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalWriteTransformer> writeTransformers() {

		int count = writeTransformersSize();
		List<ExternalWriteTransformer> transformers = new ArrayList<ExternalWriteTransformer>(count);

		for (int index = 0; index < count; index++) {
			transformers.add(buildWriteTransformer(index));
		}

		return transformers;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int writeTransformersSize() {
		return getChildrenSize(WriteTransfomer.WRITE_TRANSFORMER);
	}
}