/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalInverseJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinTable;

/**
 * The external form of a Join table, which is a child of a relationship mapping.
 *
 * @see RelationshipMapping
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
final class JoinTable extends Table
                      implements ExternalJoinTable {

	/**
	 * The node name used to store and retrieve the element encapsulated by this external form.
	 */
	static final String JOIN_TABLE = "join-table";

	/**
	 * Creates a new <code>JoinTable</code>.
	 *
	 * @param parent The parent of this external form
	 */
	JoinTable(AbstractExternalForm parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalInverseJoinColumn addInverseJoinColumn(String name) {
		InverseJoinColumn joinColumn = buildInverseJoinColumn(-1);
		joinColumn.addSelf();
		joinColumn.setName(name);
		return joinColumn;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn addJoinColumn(String name) {
		JoinColumn joinColumn = buildJoinColumn(-1);
		joinColumn.addSelf();
		joinColumn.setName(name);
		return joinColumn;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(JoinColumn.JOIN_COLUMN);
		names.add(InverseJoinColumn.INVERSE_JOIN_COLUMN);
		names.add(UniqueConstraint.UNIQUE_CONSTRAINT);
		return names;
	}

	private InverseJoinColumn buildInverseJoinColumn(int index) {
		return new InverseJoinColumn(this, index);
	}

	private JoinColumn buildJoinColumn(int index) {
		return new JoinColumn(this, JoinColumn.JOIN_COLUMN, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return JOIN_TABLE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getInverseJoinColumn(int index) {

		if (hasChild(InverseJoinColumn.INVERSE_JOIN_COLUMN, index)) {
			return buildInverseJoinColumn(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getJoinColumn(int index) {

		if (hasChild(JoinColumn.JOIN_COLUMN, index)) {
			return buildJoinColumn(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalJoinColumn> inverseJoinColumns() {

		int count = inverseJoinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = 0; index < count; index++) {
			joinColumns.add(buildJoinColumn(index));
		}

		return joinColumns;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int inverseJoinColumnsSize() {
		return getChildrenSize(InverseJoinColumn.INVERSE_JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isDefault() {
		return !(hasChildren() || hasAttributes());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalJoinColumn> joinColumns() {

		int count = joinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = 0; index < count; index++) {
			joinColumns.add(buildJoinColumn(index));
		}

		return joinColumns;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinColumnsSize() {
		return getChildrenSize(JoinColumn.JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeInverseJoinColumn(int index) {
		InverseJoinColumn joinColumn = buildInverseJoinColumn(index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.removeSelf();
	}
}