/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of an entity column.
 *
 * @see Mapping
 * @see AttributeOverride
 * @see OptimisticLocking
 * @see WriteTransfomer
 *
 * @version 2.6
 */
final class EntityColumn extends AbstractColumn
                         implements ExternalEntityColumn {

	/**
	 * The name of the element represented by this external form.
	 */
	private String elementName;

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * Creates a new <code>EntityColumn</code>.
	 *
	 * @param parent The parent of this external form
	 * @param elementName The name of the element represented by this external form
	 * @param index The position of the element within the list of children with the same type owned by the parent
	 */
	EntityColumn(AbstractExternalForm parent, String elementName, int index) {
		super(parent);
		this.index = index;
		this.elementName = elementName;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(UNIQUE);
		names.add(NULLABLE);
		names.add(INSERTABLE);
		names.add(UPDATABLE);
		names.add(COLUMN_DEFINITION);
		names.add(TABLE);
		names.add(LENGTH);
		names.add(PRECISION);
		names.add(SCALE);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void calculateInsertionIndex(Element parent, Element child, String elementName) {
		if (elementName == getElementName()) {
			index = index(parent, child, elementName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {

		if (index == -1) {
			return super.getElement();
		}

		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return elementName;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getLength() {
		return getIntegerAttribute(LENGTH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getLengthTextRange() {
		return getAttributeTextRange(LENGTH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getPrecision() {
		return getIntegerAttribute(PRECISION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPrecisionTextRange() {
		return getAttributeTextRange(PRECISION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getScale() {
		return getIntegerAttribute(SCALE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getScaleTextRange() {
		return getAttributeTextRange(SCALE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		if (index == -1) {
			super.removeSelf();
		}
		else {
			removeChild(getParent(), getElementName(), index);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLength(Integer length) {

		setAttribute(LENGTH, length);

		if (shouldRemoveSelf()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPrecision(Integer precision) {

		setAttribute(PRECISION, precision);

		if (shouldRemoveSelf()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setScale(Integer scale) {

		setAttribute(SCALE, scale);

		if (shouldRemoveSelf()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	boolean shouldRemoveEmptyElement() {
		return true;
	}

	private boolean shouldRemoveSelf() {
		return shouldRemoveEmptyElement() && !hasAnything();
	}
}