/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.FetchType;
import org.eclipse.persistence.annotations.JoinFetchType;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicMapMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalBatchFetch;
import org.eclipse.persistence.tools.mapping.orm.ExternalClassConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalCollectionTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalMappingVisitor;
import org.eclipse.persistence.tools.mapping.orm.ExternalObjectTypeConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalStructConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalTypeConverter;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form for a basic map mapping, which is a child of an entity.
 *
 * @see EmbeddableEntity
 *
 * @version 2.6
 */
final class BasicMapMapping extends ConvertibleMapping
                            implements ExternalBasicMapMapping {

	/**
	 * Creates a new <code>BasicMapMapping</code>.
	 *
	 * @param parent The parent of this external form
	 */
	BasicMapMapping(Embeddable parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void accept(ExternalMappingVisitor visitor) {
		visitor.visit(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalBatchFetch addBatchFetch() {
		BatchFetch batchFetch = buildBatchFetch();
		batchFetch.addSelf();
		return batchFetch;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalCollectionTable addCollectionTable(String name) {
		CollectionTable collectionTable = buildCollectionTable();
		collectionTable.addSelf();
		collectionTable.setName(name);
		return collectionTable;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn addKeyColumn(String columnName) {
		EntityColumn column = buildColumn(KEY_COLUMN);
		column.addSelf();
		column.setName(columnName);
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addKeyConverterString(String converter) {
		updateChildTextNode(KEY_CONVERTER, converter);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn addValueColumn(String columnName) {
		EntityColumn column = buildColumn(VALUE_COLUMN);
		column.addSelf();
		column.setName(columnName);
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(FETCH);
		names.add(ACCESS);
		return names;
	}

	private BatchFetch buildBatchFetch() {
		return new BatchFetch(this);
	}

	private CollectionTable buildCollectionTable() {
		return new CollectionTable(this);
	}

	private EntityColumn buildColumn(String elementName) {
		return new EntityColumn(this, elementName, -1);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(KEY_COLUMN);
		names.add(KEY_CONVERTER);
		names.add(VALUE_COLUMN);
		names.add(VALUE_CONVERTER);
		names.add(ClassConverter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(CollectionTable.COLLECTION_TABLE);
		names.add(JOIN_FETCH);
		names.add(BatchFetch.BATCH_FETCH);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private ClassConverter buildKeyConverter() {
		return new ClassConverter(this, -1);
	}

	private ObjectTypeConverter buildKeyObjectTypeConverter() {
		return new ObjectTypeConverter(this, -1);
	}

	private StructConverter buildKeyStructConverter() {
		return new StructConverter(this, -1);
	}

	private TypeConverter buildKeyTypeConverter() {
		return new TypeConverter(this, -1);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalBatchFetch getBatchFetch() {

		if (hasChild(BatchFetch.BATCH_FETCH)) {
			return buildBatchFetch();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalCollectionTable getCollectionTable() {

		if (hasChild(CollectionTable.COLLECTION_TABLE)) {
			return buildCollectionTable();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getConvert() {
		return getChildTextNode(VALUE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return BASIC_MAP;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FetchType getFetchType() {
		return getEnumAttribute(FETCH, FetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getFetchTypeTextRange() {
		return getAttributeTextRange(FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public JoinFetchType getJoinFetchType() {
		return getChildEnumNode(JOIN_FETCH, JoinFetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getJoinFetchTypeTextRange() {
		return getChildTextNodeTextRange(JOIN_FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getKeyColumn() {

		if (hasChild(KEY_COLUMN)) {
			return buildColumn(KEY_COLUMN);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalClassConverter getKeyConverter() {

		if (hasChild(KEY_CONVERTER)) {
			return buildKeyConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getKeyConveterTextRange() {
		return getChildTextNodeTextRange(KEY_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalObjectTypeConverter getKeyObjectTypeConverter() {

		if (hasChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER)) {
			return buildKeyObjectTypeConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStructConverter getKeyStructConverter() {

		if (hasChild(KEY_CONVERTER)) {
			return buildKeyStructConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTypeConverter getKeyTypeConverter() {

		if (hasChild(TypeConverter.TYPE_CONVERTER)) {
			return buildKeyTypeConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getValueColumn() {

		if (hasChild(VALUE_COLUMN)) {
			return buildColumn(VALUE_COLUMN);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeBatchFetch() {
		removeChild(BatchFetch.BATCH_FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeCollectionTable() {
		CollectionTable collectionTable = buildCollectionTable();
		collectionTable.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyColumn() {
		EntityColumn column = buildColumn(KEY_COLUMN);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyConverter() {
		removeChild(ClassConverter.CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyConverterString() {
		EntityColumn column = buildColumn(KEY_CONVERTER);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyObjectTypeConverter() {
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyStructConverter() {
		removeChild(StructConverter.STRUCT_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyTypeConverter() {
		removeChild(TypeConverter.TYPE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeValueColumn() {
		EntityColumn column = buildColumn(VALUE_COLUMN);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyConverter(String name, String className) {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		ClassConverter valueConverter = buildKeyConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
		valueConverter.setClassName(className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyObjectTypeConverter(String name) {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ClassConverter.CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		ObjectTypeConverter valueConverter = buildKeyObjectTypeConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyStructConverter(String name, String converter) {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(ClassConverter.CONVERTER);

		StructConverter valueConverter = buildKeyStructConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
		valueConverter.setConverter(converter);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyTypeConverter(String name) {

		removeChild(ClassConverter.CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		TypeConverter valueConverter = buildKeyTypeConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setConvert(String value) {
		updateChildTextNode(VALUE_CONVERTER, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setFetchType(FetchType type) {
		setAttribute(FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setJoinFetchType(JoinFetchType type) {
		updateChildTextNode(JOIN_FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setKeyConverter(String value) {
		updateChildTextNode(KEY_CONVERTER, value);
	}
}