/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm;

import org.eclipse.persistence.tools.utility.TextRange;

/**
 * Represents a persistent unit's meta data as defined in the ORM.xml. The root of the ORM.xml
 * (entity-mappings) supports defining persistence unit metadata. Inside the persistence unit
 * metadata properties and defaults can be configured. The intention of this interface is to
 * aggregate all of those configuration settings and defaults into a single point of access.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @see http://jcp.org/aboutJava/communityprocess/final/jsr220/index.html, persistence specificaton
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public interface ExternalPersistenceUnit extends ExternalEntityListenerHolder,
                                                 ExternalTenantDiscriminatorColumnProvider {

	/**
	 * The attribute name used to store and retrieve the cascade-persist property.
	 */
	String CASCADE_PERSIST = "cascade-persist";

	/**
	 * The attribute name used to store and retrieve the catalog property.
	 */
	String CATALOG = "catalog";

	/**
	 * The element name used to store and retrieve the delimited identifiers flag.
	 */
	String DELIMITED_IDENTIFIERS = "delimited-identifiers";

	/**
	 * The element name used to store and retrieve the description.
	 */
	String DESCRIPTION = "description";

	/**
	 * The attribute name used to store and retrieve the exclude-default-mappings property.
	 */
	String EXCLUDE_DEFAULT_MAPPINGS = "exclude-default-mappings";

	/**
	 * The attribute name used to store and retrieve the persistence-unit-defaults property.
	 */
	String PERSISTENCE_UNIT_DEFAULTS = "persistence-unit-defaults";

	/**
	 * The node name used to store and retrieve the element encapsulated by this external form.
	 */
	String PERSISTENCE_UNIT_METADATA = "persistence-unit-metadata";

	/**
	 * The attribute name used to store and retrieve the schema property.
	 */
	String SCHEMA = "schema";

	/**
	 * The attribute name used to store and retrieve the direction property.
	 */
	String XML_MAPPING_METADATA_COMPLETE = "xml-mapping-metadata-complete";

	/**
	 * Returns the {@link TextRange} for the default cascade option value.
	 */
	TextRange getCascadePersistByDefaultTextRange();

	/**
	 * Returns the default access type to be used for entities that comprise this persistence unit.
	 */
	AccessType getDefaultAccessType();

	/**
	 * Returns the {@link TextRange} for the default access type value.
	 */
	TextRange getDefaultAccessTypeTextRange();

	/**
	 * Returns the default catalog name to be used in the persistence unit.
	 */
	String getDefaultCatalogName();

	/**
	 * Returns the {@link TextRange} of the default catalog name value.
	 */
	TextRange getDefaultCatalogNameTextRange();

	/**
	 * Returns the default description to be used in the persistence unit.
	 */
	String getDefaultDescription();

	/**
	 * Returns the default schema name to be used in the persistence unit.
	 */
	String getDefaultSchemaName();

	/**
	 * Returns the {@link TextRange} of the default schema name value.
	 */
	TextRange getDefaultSchemaNameTextRange();

	/**
	 * Returns the description for the persistence unit.
	 */
	String getDescription();

	/**
	 * Indicates whether this external persistence unit declaration is in it's default state and has
	 * not child values defined.
	 */
	boolean isDefault();

	/**
	 * Indicates whether this external persistence unit defaults should use delimited identifiers.
	 */
	boolean isDelimitedIdentifers();

	/**
	 * Indicates whether the default mappings should be excluded for the persistence unit settings
	 * and defaults.
	 */
	boolean isExcludeDefaultMappings();

	/**
	 * Indicates whether the mapping meta data is complete for the persistence unit settings and defaults.
	 */
	boolean isMappingMetaDataComplete();

	/**
	 * Sets the access default type.
	 */
	void setDefaultAccessType(AccessType type);

	/**
	 * Sets the default catalog name to be used in the persistence unit.
	 */
	void setDefaultCatalogName(String catalogName);

	/**
	 * Sets the default description to be used in the persistence unit.
	 */
	void setDefaultDescription(String description);

	/**
	 * Sets the default schema name to be used in the persistence unit.
	 */
	void setDefaultSchemaName(String schemaName);

	/**
	 * Indicates whether this external persistence unit defaults should use delimited identifiers.
	 */
	void setDelimitedIdentifers(boolean delimitIdentifiers);

	/**
	 * Sets the description.
	 */
	void setDescription(String description);

	/**
	 * Sets whether the default mappings should be excluded
	 */
	void setExcludeDefaultMappings(boolean excludeDefaultMappings);

	/**
	 * Sets whether the mapping meta data is complete
	 */
	void setMappingMetaDataComplete(boolean metaDataComplete);

	/**
	 * Sets the default cascade option to use for relationships.
	 */
	void setUsesCascadePersistByDefault(boolean cascadePersistByDefault);

	/**
	 * Indicates the default cascade option to use for relationships of entities comprising this
	 * persistence unit.
	 */
	boolean usesCascadePersistByDefault();
}