/*******************************************************************************
 * Copyright (c) 2009, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.reference;

import java.io.Serializable;
import org.eclipse.persistence.tools.utility.ObjectTools;

/**
 * Provide a container for passing an object that can be changed by the recipient.
 *
 * @see SynchronizedObject
 */
public class SimpleObjectReference<V>
	implements ModifiableObjectReference<V>, Cloneable, Serializable
{
	/** Backing value. */
	private volatile V value;

	private static final long serialVersionUID = 1L;


	// ********** constructors **********

	/**
	 * Create an object reference with the specified initial value.
	 */
	public SimpleObjectReference(V value) {
		super();
		this.value = value;
	}

	/**
	 * Create an object reference with an initial value of
	 * <code>null</code>.
	 */
	public SimpleObjectReference() {
		this(null);
	}


	// ********** value **********

	@Override
	public V getValue() {
		return this.value;
	}

	@Override
	public boolean valueEquals(Object object) {
		return ObjectTools.equals(this.value, object);
	}

	@Override
	public boolean valueNotEqual(Object object) {
		return ObjectTools.notEquals(this.value, object);
	}

	@Override
	public boolean isNull() {
		return this.value == null;
	}

	@Override
	public boolean isNotNull() {
		return this.value != null;
	}

	@Override
	public V setValue(V value) {
		V old = this.value;
		this.value = value;
		return old;
	}

	@Override
	public V setNull() {
		return this.setValue(null);
	}


	// ********** standard methods **********

	@Override
	public SimpleObjectReference<V> clone() {
		try {
			@SuppressWarnings("unchecked")
			SimpleObjectReference<V> clone = (SimpleObjectReference<V>) super.clone();
			return clone;
		} catch (CloneNotSupportedException ex) {
			throw new InternalError();
		}
	}

	@Override
	public String toString() {
		return '[' + String.valueOf(this.value) + ']';
	}
}