/*******************************************************************************
 * Copyright (c) 2005, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.iterator;

import java.util.List;
import java.util.ListIterator;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.transformer.Transformer;

/**
 * A <code>TransformationListIterator</code> wraps another {@link ListIterator}
 * and transforms its results for client consumption. To use, supply a
 * {@link Transformer} or subclass <code>TransformationIterator</code>
 * and override the {@link #transform(Object)} method.
 * <p>
 * The methods {@link #set(Object)} and {@link #add(Object)}
 * are left unsupported in this class.
 *
 * @param <E1> input: the type of elements to be transformed
 * @param <E2> output: the type of elements returned by the iterator
 *
 * @see org.eclipse.jpt.common.utility.internal.iterable.TransformationListIterable
 */
public class TransformationListIterator<E1, E2>
	implements ListIterator<E2>
{
	private final ListIterator<? extends E1> listIterator;
	private final Transformer<E1, ? extends E2> transformer;


	/**
	 * Construct an iterator with the specified list
	 * and a disabled transformer.
	 * Use this constructor if you want to override the
	 * {@link #transform(Object)} method instead of building
	 * a {@link Transformer}.
	 */
	public TransformationListIterator(List<? extends E1> list) {
		this(list.listIterator());
	}

	/**
	 * Construct an iterator with the specified nested listed iterator
	 * and a disabled transformer.
	 * Use this constructor if you want to override the
	 * {@link #transform(Object)} method instead of building
	 * a {@link Transformer}.
	 */
	public TransformationListIterator(ListIterator<? extends E1> listIterator) {
		this(listIterator, Transformer.Disabled.<E1, E2>instance());
	}

	/**
	 * Construct an iterator with the specified list
	 * and a disabled transformer.
	 * Use this constructor if you want to override the
	 * {@link #transform(Object)} method instead of building
	 * a {@link Transformer}.
	 */
	public TransformationListIterator(ListIterable<? extends E1> listIterable) {
		this(listIterable.iterator());
	}

	/**
	 * Construct an iterator with the specified list and transformer.
	 */
	public TransformationListIterator(List<? extends E1> list, Transformer<E1, ? extends E2> transformer) {
		this(list.listIterator(), transformer);
	}

	/**
	 * Construct an iterator with the specified list and transformer.
	 */
	public TransformationListIterator(ListIterable<? extends E1> listIterable, Transformer<E1, ? extends E2> transformer) {
		this(listIterable.iterator(), transformer);
	}

	/**
	 * Construct an iterator with the specified nested iterator
	 * and transformer.
	 */
	public TransformationListIterator(ListIterator<? extends E1> listIterator, Transformer<E1, ? extends E2> transformer) {
		super();
		if ((listIterator == null) || (transformer == null)) {
			throw new NullPointerException();
		}
		this.listIterator = listIterator;
		this.transformer = transformer;
	}

	@Override
	public boolean hasNext() {
		// delegate to the nested iterator
		return this.listIterator.hasNext();
	}

	@Override
	public E2 next() {
		// transform the object returned by the nested iterator before returning it
		return this.transform(this.listIterator.next());
	}

	@Override
	public int nextIndex() {
		// delegate to the nested iterator
		return this.listIterator.nextIndex();
	}

	@Override
	public boolean hasPrevious() {
		// delegate to the nested iterator
		return this.listIterator.hasPrevious();
	}

	@Override
	public E2 previous() {
		// transform the object returned by the nested iterator before returning it
		return this.transform(this.listIterator.previous());
	}

	@Override
	public int previousIndex() {
		// delegate to the nested iterator
		return this.listIterator.previousIndex();
	}

	@Override
	public void add(E2 o) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void set(E2 o) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void remove() {
		// delegate to the nested iterator
		this.listIterator.remove();
	}

	/**
	 * Transform the specified object and return the result.
	 */
	protected E2 transform(E1 next) {
		return this.transformer.transform(next);
	}

	@Override
	public String toString() {
		return ObjectTools.toString(this, this.listIterator);
	}
}