/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalNoSqlJoinField;
import org.eclipse.persistence.tools.mapping.orm.ExternalOneToOneMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalPrimaryKeyJoinColumn;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form for a 1:1 mapping, which is a child of an entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class OneToOneMapping extends RelationshipMapping
                            implements ExternalOneToOneMapping {

	/**
	 * The attribute name used to store and retrieve the id property.
	 */
	static final String ID = "id";

	/**
	 * The attribute name used to store and retrieve the mapped-by property.
	 */
	static final String MAPPED_BY = "mapped-by";

	/**
	 * The attribute name used to store and retrieve the maps-id property.
	 */
	static final String MAPS_ID = "maps-id";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String ONE_TO_ONE = "one-to-one";

	/**
	 * The attribute name used to store and retrieve the orphan-removal property.
	 */
	static final String ORPHAN_REMOVAL = "orphan-removal";

	/**
	 * The attribute name used to store and retrieve the private-owned property.
	 */
	static final String PRIVATE_OWNED = "private-owned";

	/**
	 * Creates a new <code>OneToOneMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	OneToOneMapping(EmbeddableEntity parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addJoinColumn(int index, String name) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.addSelf();
		joinColumn.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addJoinField(int index, String name) {
		ExternalNoSqlJoinField joinField = buildJoinField(index);
		joinField.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addPrimaryKeyJoinColumn(int index, String name) {
		PrimaryKeyJoinColumn pkJoinColumn = buildPrimaryKeyJoinColumn(index);
		pkJoinColumn.addSelf();
		pkJoinColumn.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(TARGET_ENTITY);
		names.add(FETCH);
		names.add(OPTIONAL);
		names.add(ACCESS);
		names.add(MAPPED_BY);
		names.add(ORPHAN_REMOVAL);
		names.add(MAPS_ID);
		names.add(ID);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(PrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN);
		names.add(JoinColumn.JOIN_COLUMN);
		names.add(JoinTable.JOIN_TABLE);
		names.add(CASCADE);
		names.add(PRIVATE_OWNED);
		names.add(JOIN_FETCH);
		names.add(BatchFetch.BATCH_FETCH);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private JoinColumn buildJoinColumn(int index) {
		return new JoinColumn(this, index);
	}

	private ExternalNoSqlJoinField buildJoinField(int index) {
		return new NoSqlJoinField(this, index);
	}

	private PrimaryKeyJoinColumn buildPrimaryKeyJoinColumn(int index) {
		return new PrimaryKeyJoinColumn(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ONE_TO_ONE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean getId() {
		return getBooleanAttribute(ID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getJoinColumn(int index) {

		Element element = getChild(JoinColumn.JOIN_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildJoinColumn(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMappedByMappingName() {
		return getAttribute(MAPPED_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapsId() {
		return getAttribute(MAPS_ID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalPrimaryKeyJoinColumn getPrimaryKeyJoinColumn(int index) {

		Element element = getChild(PrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildPrimaryKeyJoinColumn(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isOrphanRemoval() {
		return getBooleanAttribute(ORPHAN_REMOVAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isPrivateOwned() {
		return hasChild(PRIVATE_OWNED);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalJoinColumn> joinColumns() {

		int count = joinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalJoinColumn joinColumn = buildJoinColumn(index);
			joinColumns.add(0, joinColumn);
		}

		return new ListListIterable<ExternalJoinColumn>(joinColumns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinColumnsSize() {
		return getChildrenSize(JoinColumn.JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalNoSqlJoinField> joinFields() {

		int count = joinFieldSize();
		List<ExternalNoSqlJoinField> joinFields = new ArrayList<ExternalNoSqlJoinField>(count);

		for (int index = count; --index >= 0;) {
			ExternalNoSqlJoinField joinField = buildJoinField(index);
			joinFields.add(0, joinField);
		}

		return new ListListIterable<ExternalNoSqlJoinField>(joinFields);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinFieldSize() {
		return getChildrenSize(NoSqlJoinField.JOIN_FIELD);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalPrimaryKeyJoinColumn> primaryKeyJoinColumns() {

		int count = primaryKeyJoinColumnsSize();
		List<ExternalPrimaryKeyJoinColumn> pkJoinColumns = new ArrayList<ExternalPrimaryKeyJoinColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalPrimaryKeyJoinColumn pkJoinColumn = buildPrimaryKeyJoinColumn(index);
			pkJoinColumns.add(0, pkJoinColumn);
		}

		return new ListListIterable<ExternalPrimaryKeyJoinColumn>(pkJoinColumns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int primaryKeyJoinColumnsSize() {
		return getChildrenSize(PrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinField(int index) {
		removeChild(NoSqlJoinField.JOIN_FIELD, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removePrimaryKeyJoinColumn(int index) {
		PrimaryKeyJoinColumn pkJoinColumn = buildPrimaryKeyJoinColumn(index);
		pkJoinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setId(Boolean id) {
		setAttribute(ID, id);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMappedByMappingName(String name) {
		setAttribute(MAPPED_BY, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapsId(String mapsId) {
		setAttribute(MAPS_ID, mapsId);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOrphanRemoval(Boolean removeOrphans) {
		setAttribute(ORPHAN_REMOVAL, removeOrphans);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPrivateOwned(Boolean privateOwned) {
		if (privateOwned == Boolean.TRUE) {
			addChild(PRIVATE_OWNED);
		}
		else {
			removeChild(PRIVATE_OWNED);
		}
	}
}