/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.annotations.CacheCoordinationType;
import org.eclipse.persistence.annotations.CacheType;
import org.eclipse.persistence.annotations.DatabaseChangeNotificationType;
import org.eclipse.persistence.config.CacheIsolationType;
import org.eclipse.persistence.tools.mapping.orm.ExternalCache;
import org.eclipse.persistence.tools.mapping.orm.ExternalTimeOfDay;

/**
 * The external form of a cache, which is a child of a mapped superclass entity.
 *
 * @see MappedSuperClassEntity
 * 
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class Cache extends AbstractExternalForm
                  implements ExternalCache {

	/**
	 * The attribute name used to store and retrieve the always-refresh property.
	 */
	static final String ALWAYS_REFRESH = "always-refresh";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String CACHE = "cache";

	/**
	 * The attribute name used to store and retrieve the coordination-type property.
	 */
	static final String COORDINATION_TYPE = "coordination-type";
	
	/**
	 * The attribute name used to store and retrieve the database-change-notification-type property.
	 */
	static final String DATABASE_CHANGE_NOTIFICATION_TYPE = "database-change-notification-type";

	/**
	 * The attribute name used to store and retrieve the disable-hits property.
	 */
	static final String DISABLE_HITS = "disable-hits";

	/**
	 * The element name used to store and retrieve the expiry child text node.
	 */
	static final String EXPIRY = "expiry";
	
	/**
	 * The attribute name used to store and retrieve the isolation property.
	 */
	static final String ISOLATION = "isolation";

	/**
	 * The attribute name used to store and retrieve the refresh-only-if-newer property.
	 */
	static final String REFRESH_ONLY_IF_NEWER = "refresh-only-if-newer";

	/**
	 * The attribute name used to store and retrieve the shared property.
	 */
	static final String SHARED = "shared";

	/**
	 * The attribute name used to store and retrieve the size property.
	 */
	static final String SIZE = "size";

	/**
	 * The attribute name used to store and retrieve the type property.
	 */
	static final String TYPE = "type";

	/**
	 * Creates a new <code>Cache</code>.
	 *
	 * @param parent The parent of this external form
	 */
	Cache(MappedSuperClassEntity parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addExpiryTimeOfDay() {
		TimeOfDay timeOfDay = buildTimeOfDay();
		timeOfDay.addSelf();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(SIZE);
		names.add(SHARED);
		names.add(TYPE);
		names.add(ALWAYS_REFRESH);
		names.add(REFRESH_ONLY_IF_NEWER);
		names.add(DISABLE_HITS);
		names.add(COORDINATION_TYPE);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(EXPIRY);
		names.add(TimeOfDay.EXPIRY_TIME_OF_DAY);
		return names;
	}

	private TimeOfDay buildTimeOfDay() {
		return new TimeOfDay(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CacheCoordinationType getCoordinationType() {
		return getEnumAttribute(COORDINATION_TYPE, CacheCoordinationType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public DatabaseChangeNotificationType getDatabaseChangeNotificationType() {
		return getEnumAttribute(DATABASE_CHANGE_NOTIFICATION_TYPE, DatabaseChangeNotificationType.class);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return CACHE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTimeOfDay getExpiryTimeOfDay() {

		if (hasChild(TimeOfDay.EXPIRY_TIME_OF_DAY)) {
			return buildTimeOfDay();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getExpiryTimeToLive() {
		return getChildIntegerNode(EXPIRY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CacheIsolationType getIsolationType() {
		return getEnumAttribute(ISOLATION, CacheIsolationType.class);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getSize() {
		return getIntegerAttribute(SIZE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CacheType getType() {
		return getEnumAttribute(TYPE, CacheType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isAlwaysRefresh() {
		return getBooleanAttribute(ALWAYS_REFRESH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isDisableHits() {
		return getBooleanAttribute(DISABLE_HITS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isRefreshOnlyIfNewer() {
		return getBooleanAttribute(REFRESH_ONLY_IF_NEWER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isShared() {
		return getBooleanAttribute(SHARED);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeExpiryTimeOfDay() {

		TimeOfDay timeOfDay = buildTimeOfDay();
		timeOfDay.removeSelf();

		if (!hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAlwaysRefresh(Boolean alwaysRefresh) {

		setAttribute(ALWAYS_REFRESH, alwaysRefresh);

		if (!hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setCoordinationType(CacheCoordinationType type) {

		setAttribute(COORDINATION_TYPE, type);

		if (!hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}
	
	@Override
	public void setDatabaseChangeNotificationType(DatabaseChangeNotificationType type) {
		setAttribute(DATABASE_CHANGE_NOTIFICATION_TYPE, type);
		
		if (!hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDisableHits(Boolean disableHits) {

		setAttribute(DISABLE_HITS, disableHits);

		if (!hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setExpiryTimeToLive(Integer timeToLive) {

		updateChildTextNode(EXPIRY, timeToLive);

		if (!hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}
	
	@Override
	public void setIsolationType(CacheIsolationType isolationType) {
		setAttribute(ISOLATION, isolationType);

		if (!hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setRefreshOnlyIfNewer(Boolean refreshOnlyIfNewer) {

		setAttribute(REFRESH_ONLY_IF_NEWER, refreshOnlyIfNewer);

		if (!hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setShared(Boolean shared) {

		setAttribute(SHARED, shared);

		if (!hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSize(Integer size) {

		setAttribute(SIZE, size);

		if (!hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setType(CacheType type) {

		setAttribute(TYPE, type);

		if (!hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}
}