/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.persistence;

import java.util.List;
import java.util.ListIterator;
import javax.persistence.ValidationMode;
import org.eclipse.persistence.tools.mapping.orm.ExternalForm;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;

/**
 * Interface defining one persistence unit in the persistence.xml file
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.5
 * @author Les Davis
 * @author John Bracken
 */
public interface ExternalPersistenceUnit extends ExternalForm {

	/**
	 * Sets the named property's value.
	 *
	 * @param index The position of the property to add within the list of properties already defined
	 * @param name The key of the property
	 * @param value The value of the property
	 */
	ExternalProperty addProperty(int index, String name, String value);

	/**
	 * Adds a new named property.
	 *
	 * @param name The key of the property
	 * @param value The value of the property
	 */
	ExternalProperty addProperty(String name, String value);

	/**
	 * Returns the cache type for this persistence unit.
	 */
	CacheType getCachType();

	/**
	 * Returns a text description of the persistence unit.
	 */
	String getDescription();

	/**
	 * Returns the position of this persistence unit.
	 */
	int getIndex();

	/**
	 * Returns the class name of the JDBC driver used, fully qualified by its package name.
	 */
	String getJdbcDriver();

	/**
	 * Returns the password used to connect to the data source via a JDBC connection.
	 */
	String getJdbcPassword();

	/**
	 * Returns the JDBC connection URL required by the JDBC driver, which is used to connect to the
	 * data source.
	 */
	String getJdbcUrl();

	/**
	 * Returns the user name used to connect to the data source via a JDBC connection.
	 */
	String getJdbcUser();

	/**
	 * Returns the JTA data source name for the persistence unit.
	 */
	String getJtaDatasource();

	/**
	 * Returns the name of the persistence unit.
	 */
	String getName();

	/**
	 * Returns the non-JTA data source name for the persistence unit.
	 */
	String getNonJtaDatasource();

	/**
	 * Returns the pessimistic lock timeout in milliseconds for the persistence unit.
	 */
	Integer getPessimisticLockTimeout();

	/**
	 * Retrieves the external properties with the given name.
	 *
	 * @param name The name of the properties to retrieve
	 * @return The list of properties with the given name as its key. A list is returned since it's
	 * possible to have more than one properties with the same key
	 */
	List<ExternalProperty> getProperties(String name);

	/**
	 * Returns the a persistence unit property with the given name and value. Callers of this method
	 * assume that only one property exists with the given name. If multiple properties exist,
	 * acquire the property by name and value.
	 *
	 * @param index The position of the property to retrieve within the list of properties
	 * @return The external form of the property
	 */
	ExternalProperty getProperty(int index);

	/**
	 * Retrieves the external property, if one was found, with the given name.
	 *
	 * @param name The name of the key used to retrieve the property
	 * @return The property with the given name or <code>null</code> if none could be found
	 */
	ExternalProperty getProperty(String name);

	/**
	 * Retrieves the property at the given index from the list of properties with the given name.
	 *
	 * @param name The name used to retrieve the property at the given index within the list of
	 * properties with that same name
	 * @param index The index of the property to retrieve
	 * @return The property at the given index with the given name
	 */
	ExternalProperty getProperty(String name, int index);

	/**
	 * Retrieves the external property, if one was found, with the given name and value.
	 *
	 * @param name The name used to retrieve the property
	 * @param value The value used to retrieve the property
	 * @return The external property matching the given name and value if one could be found
	 */
	ExternalProperty getProperty(String name, String value);

	/**
	 * Returns the class name of the persistence provider for this persistence unit.
	 */
	String getProvider();

	/**
	 * Returns the query timeout in milliseconds for this persistence unit.
	 */
	Integer getQueryTimeout();

	/**
	 * Returns the transaction type for this persistence unit.
	 */
	TransactionType getTransactionType();

	/**
	 * Returns the validation mode for this persistence unit.
	 */
	ValidationMode getValidationMode();

	/**
	 * Returns a collection of class names that should be included in the persistence unit.
	 */
	ListIterable<String> includedClasses();

	/**
	 * Returns the count of all the class names that should be included in the persistence unit.
	 */
	int includedClassesSize();

	/**
	 * Returns a collection of jar files that should be included in the persistence unit.
	 */
	ListIterable<String> jarFiles();

	/**
	 * Returns the count of all the Jar file entries.
	 */
	int jarFilesSize();

	/**
	 * Returns a collection of mapping files, these are orm.xml files that should be included in the
	 * persistence unit.
	 */
	ListIterable<String> mappingFiles();

	/**
	 * Returns the count of all the JPA Mapping file entries.
	 */
	int mappingFilesSize();

	/**
	 * Returns the properties associated with this persistence unit.
	 */
	ListIterable<ExternalProperty> properties();

	/**
	 * Returns the count of properties defined for this persistence unit.
	 */
	int propertiesSize();

	/**
	 * Returns the count of properties defined for this persistence unit that has the given name.
	 *
	 * @param name The name used to determine the count of properties where their key matches that name
	 * @return The count of properties defined with the given name
	 */
	int propertiesSize(String name);

	/**
	 * Removes the given property from this unit.
	 *
	 * @param index The position of the external property to remove
	 */
	void removeProperty(int index);

	/**
	 * Removes the first occurrence of the given property from this unit.
	 *
	 * @param name The name of the external property to remove
	 * @param value The value of the external property to remove
	 */
	void removeProperty(String name, String value);

	/**
	 * Sets the cache type for this persistence unit.
	 */
	void setCacheType(CacheType type);

	/**
	 * Sets the description for the persistence unit.
	 */
	void setDescription(String description);

	/**
	 * Sets whether the persistence unit should exclude annotated classes in root of this persistence unit.
	 */
	void setExcludeUnlistedClasses(boolean excludeUnlistedClasses);

	/**
	 * Adds a included class name.
	 */
	void setIncludedClasses(ListIterator<String> classes);

	/**
	 * Updates the list of Jar file.
	 */
	void setJarFiles(ListIterator<String> jarFiles);

	/**
	 * Sets the JTA data source name for the persistence unit.
	 */
	void setJtaDatasource(String name);

	/**
	 * Adds a mapping file.
	 */
	void setMappingFiles(ListIterator<String> mappingFiles);

	/**
	 * Sets the name of the persistence unit.
	 */
	void setName(String name);

	/**
	 * Sets the non-JTA data source name for the persistence unit.
	 */
	void setNonJtaDatasource(String name);

	/**
	 * Sets the pessimistic lock timeout in milliseconds for this persistence unit.
	 */
	void setPessimisticLockTimeout(Integer length);

	/**
	 * Sets the class name of the persistence provider for this persistence unit.
	 */
	void setProvider(String name);

	/**
	 * Sets the query timeout in milliseconds for this persistence unit.
	 */
	void setQueryTimeout(Integer length);

	/**
	 * Sets the transaction type for this persistence unit.
	 */
	void setTransactionType(TransactionType type);

	/**
	 * Sets the validation mode type for this persistence unit.
	 */
	void setValidationMode(ValidationMode type);

	/**
	 * Returns whether the persistence unit should exclude annotated classes in
	 * root of the persistence unit
	 */
	boolean shouldExcludeUnlistedClasses();

	/**
	 * Enumeration of shared cache types for a persistence unit.
	 */
	enum CacheType {
		ALL,
		DISABLE_SELECTIVE,
		ENABLE_SELECTIVE,
		NONE,
		UNSPECIFIED
	}

	/**
	 * Enumeration of the possible persistence types for a persistence unit.
	 */
	enum TransactionType {
		JTA,
		RESOURCE_LOCAL
	}
}