/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.annotations.CacheKeyType;
import org.eclipse.persistence.annotations.IdValidation;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalPrimaryKey;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of a primary key, which is a child of an entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class PrimaryKey extends AbstractExternalForm
                       implements ExternalPrimaryKey {

	/**
	 * The attribute name used to store and retrieve the cache-key-type of the property.
	 */
	static final String CACHE_KEY_TYPE = "cache-key-type";

	/**
	 * The attribute name used to store and retrieve the id-validation of the property.
	 */
	static final String ID_VALIDATION = "id-validation";

	/**
	 * The attribute name used to store and retrieve the primary-key of the property.
	 */
	static final String PRIMARY_KEY = "primary-key";

	/**
	 * Creates a new <code>PrimaryKey</code>.
	 *
	 * @param parent The parent of this external form
	 */
	PrimaryKey(MappedSuperClassEntity parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addColumn(int index, String columnName) {
		EntityColumn column = buildColumn(index);
		column.addSelf();
		column.setName(columnName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(ID_VALIDATION);
		return names;
	}

	private EntityColumn buildColumn(int index) {
		return new EntityColumn(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(EntityColumn.COLUMN);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalEntityColumn> columns() {

		int count = columnsSize();
		List<ExternalEntityColumn> columns = new ArrayList<ExternalEntityColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalEntityColumn column = buildColumn(index);
			columns.add(0, column);
		}

		return new ListListIterable<ExternalEntityColumn>(columns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int columnsSize() {
		return getChildrenSize(EntityColumn.COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CacheKeyType getCacheKey() {
		return getEnumAttribute(CACHE_KEY_TYPE, CacheKeyType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getColumn(int index) {

		Element element = getChild(EntityColumn.COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildColumn(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return PRIMARY_KEY;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IdValidation getValidation() {
		return getEnumAttribute(ID_VALIDATION, IdValidation.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAllColumns() {
		removeChildren(EntityColumn.COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeColumn(int index) {
		EntityColumn column = buildColumn(index);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setCacheKey(CacheKeyType cacheKeyType) {
		setAttribute(CACHE_KEY_TYPE, cacheKeyType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setValidation(IdValidation validation) {
		setAttribute(ID_VALIDATION, validation);
	}
}