/*******************************************************************************
 * Copyright (c) 2009, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.collection;

import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.iterator.EmptyIterator;
import org.eclipse.persistence.tools.utility.iterator.EmptyListIterator;

/**
 * A "null" list is a bit different from an "empty" list: it allows clients to
 * add/remove elements to/from it but never changes. This is useful
 * for passing to methods that require a "collecting parameter" but the
 * client will ignore the resulting "collection".
 */
@SuppressWarnings("nls")
public final class NullList<E>
	implements List<E>, Serializable
{
	// singleton
	@SuppressWarnings("rawtypes")
	private static final NullList INSTANCE = new NullList();

	/**
	 * Return the singleton.
	 */
	@SuppressWarnings("unchecked")
	public static <E> List<E> instance() {
		return INSTANCE;
	}

	/**
	 * Ensure single instance.
	 */
	private NullList() {
		super();
	}

	@Override
	public boolean add(E o) {
		return false;  // the list did not change
	}

	@Override
	public void add(int index, E element) {
		// ignore
	}

	@Override
	public boolean addAll(Collection<? extends E> c) {
		return false;  // the list did not change
	}

	@Override
	public boolean addAll(int index, Collection<? extends E> c) {
		return false;  // the list did not change
	}

	@Override
	public void clear() {
		// ignore
	}

	@Override
	public boolean contains(Object o) {
		return false;
	}

	@Override
	public boolean containsAll(Collection<?> c) {
		return c.isEmpty();
	}

	@Override
	public E get(int index) {
		throw new IndexOutOfBoundsException("Index: " + index + ", Size: 0");
	}

	@Override
	public int indexOf(Object o) {
		return -1;
	}

	@Override
	public boolean isEmpty() {
		return true;
	}

	@Override
	public Iterator<E> iterator() {
		return EmptyIterator.instance();
	}

	@Override
	public int lastIndexOf(Object o) {
		return -1;
	}

	@Override
	public ListIterator<E> listIterator() {
		return EmptyListIterator.instance();
	}

	@Override
	public ListIterator<E> listIterator(int index) {
		return EmptyListIterator.instance();
	}

	@Override
	public boolean remove(Object o) {
		return false;  // the list did not change
	}

	@Override
	public E remove(int index) {
		throw new IndexOutOfBoundsException("Index: " + index + ", Size: 0");
	}

	@Override
	public boolean removeAll(Collection<?> c) {
		return false;  // the list did not change
	}

	@Override
	public boolean retainAll(Collection<?> c) {
		return false;  // the list did not change
	}

	@Override
	public E set(int index, E element) {
		throw new IndexOutOfBoundsException("Index: " + index + ", Size: 0");
	}

	@Override
	public int size() {
		return 0;
	}

	@Override
	public List<E> subList(int fromIndex, int toIndex) {
		return this;
	}

	@Override
	public Object[] toArray() {
		return ObjectTools.EMPTY_OBJECT_ARRAY;
	}

	@Override
	public <T> T[] toArray(T[] a) {
		return a;
	}

	@Override
	public String toString() {
		return this.getClass().getSimpleName();
	}

	private static final long serialVersionUID = 1L;
	private Object readResolve() {
		// replace this object with the singleton
		return INSTANCE;
	}
}