/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.node;

import java.io.Serializable;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.reference.SynchronizedBoolean;

/**
 * This implementation of the {@link org.eclipse.jpt.common.utility.node.Node.Validator}
 * interface implements the
 * pause/resume portion of the protocol, but delegates the actual
 * validation to a "pluggable" delegate.
 */
@SuppressWarnings("nls")
public class PluggableValidator
	implements Node.Validator
{
	private boolean pause;
	private boolean validateOnResume;
	private final Delegate delegate;


	/**
	 * Convenience factory method.
	 */
	public static Node.Validator buildAsynchronousValidator(SynchronizedBoolean validateFlag) {
		return new PluggableValidator(new AsynchronousValidator(validateFlag));
	}

	/**
	 * Convenience factory method.
	 */
	public static Node.Validator buildSynchronousValidator(Node node) {
		return new PluggableValidator(new SynchronousValidator(node));
	}

	/**
	 * Construct a validator with the specified delegate.
	 */
	public PluggableValidator(Delegate delegate) {
		super();
		if (delegate == null) {
			throw new NullPointerException();
		}
		this.pause = false;
		this.validateOnResume = false;
		this.delegate = delegate;
	}

	@Override
	public synchronized void validate() {
		if (this.pause) {
			this.validateOnResume = true;
		} else {
			this.delegate.validate();
		}
	}

	@Override
	public synchronized void pause() {
		if (this.pause) {
			throw new IllegalStateException("already paused");
		}
		this.pause = true;
	}

	@Override
	public synchronized void resume() {
		if ( ! this.pause) {
			throw new IllegalStateException("not paused");
		}
		this.pause = false;
		// validate any changes that occurred while the validation was paused
		if (this.validateOnResume) {
			this.validateOnResume = false;
			this.delegate.validate();
		}
	}

	@Override
	public String toString() {
		return ObjectTools.toString(this, this.delegate);
	}


	// ********** member interface **********

	/**
	 * Interface implemented by any delegates of a pluggable validator.
	 */
	public interface Delegate {

		/**
		 * The validator is not "paused" - perform the appropriate validation.
		 */
		void validate();


		/**
		 * This delegate does nothing.
		 */
		final class Null
			implements Delegate, Serializable
		{
			public static final Delegate INSTANCE = new Null();
			public static Delegate instance() {
				return INSTANCE;
			}
			// ensure single instance
			private Null() {
				super();
			}
			@Override
			public void validate() {
				// do nothing
			}
			@Override
			public String toString() {
				return ObjectTools.singletonToString(this);
			}
			private static final long serialVersionUID = 1L;
			private Object readResolve() {
				// replace this object with the singleton
				return INSTANCE;
			}
		}
	}
}