/*******************************************************************************
 * Copyright (c) 2012, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.command;

/**
 * Calls to {@link #waitToExecute(Command)} will suspend the current thread
 * until the command executor is {@link #start() started} and any previously-
 * dispatched commands have executed.
 *
 * @see AbstractQueueingCommandExecutor
 */
public class QueueingExtendedCommandExecutor
	extends AbstractQueueingCommandExecutor<StatefulExtendedCommandExecutor>
	implements StatefulExtendedCommandExecutor
{
	public QueueingExtendedCommandExecutor() {
		this(ExtendedCommandExecutor.Default.instance());
	}

	public QueueingExtendedCommandExecutor(ExtendedCommandExecutor commandExecutor) {
		this(new SimpleStatefulExtendedCommandExecutor(commandExecutor));
	}

	public QueueingExtendedCommandExecutor(StatefulExtendedCommandExecutor commandExecutor) {
		super(commandExecutor);
	}

	/**
	 * Suspend the current thread until the command executor is
	 * {@link #start() started}.
	 */
	@Override
	public void waitToExecute(Command command) throws InterruptedException {
		SynchronizingCommand syncCommand = new SynchronizingCommand();

		this.execute(syncCommand);  // put the sync command on the queue and wait until it has executed

		try {
			syncCommand.waitForExecution();
			this.commandExecutor.waitToExecute(command);
		} finally {
			syncCommand.release();
		}
	}

	/**
	 * Wait until all the commands currently on the queue have been executed,
	 * then execute the specified command. Suspend the command-executing thread
	 * while we execute the specified command.
	 */
	@Override
	public boolean waitToExecute(Command command, long timeout) throws InterruptedException {
		if (timeout == 0L) {
			this.waitToExecute(command);
			return true;
		}

		long stop = System.currentTimeMillis() + timeout;
		SynchronizingCommand syncCommand = new SynchronizingCommand();

		this.execute(syncCommand);  // dispatch the sync command to the other thread

		try {
			if (syncCommand.waitForExecution(timeout)) {
				// adjust the time
				timeout = stop - System.currentTimeMillis();
				return (timeout > 0) && this.commandExecutor.waitToExecute(command, timeout);
			}
			return false;
		} finally {
			syncCommand.release();
		}
	}
}