/*******************************************************************************
 * Copyright (c) 2009, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.Collection;
import org.eclipse.persistence.tools.utility.iterable.IterableTools;

/**
 * Provide a way for multiple exceptions to be packaged and reported.
 */
@SuppressWarnings("nls")
public class CompositeException
	extends RuntimeException
{
	private final Throwable[] exceptions;
	private static final long serialVersionUID = 1L;


	/**
	 * The specified exceptions list must not be empty.
	 */
	public CompositeException(Collection<Throwable> exceptions) {
		this(exceptions.toArray(new Throwable[exceptions.size()]));
	}

	/**
	 * The specified exceptions list must not be empty.
	 */
	public CompositeException(Throwable... exceptions) {
		// provide a list of the nested exceptions and
		// grab the first exception and make it the "cause"
		super(buildMessage(exceptions));
		this.exceptions = exceptions;
	}

	public Iterable<Throwable> getExceptions() {
		return IterableTools.iterable(this.exceptions);
	}

	private static String buildMessage(Throwable[] exceptions) {
		StringBuilder sb = new StringBuilder();
		sb.append(exceptions.length);
		sb.append(" exceptions: ");
		sb.append('[');
		for (Throwable ex : exceptions) {
			sb.append(ex.getClass().getSimpleName());
			sb.append(", ");
		}
		sb.setLength(sb.length() - 2);  // chop off trailing comma
		sb.append(']');
		return sb.toString();
	}

	@Override
	public void printStackTrace(PrintStream s) {
		synchronized (s) {
			s.println(this);
			for (StackTraceElement element : this.getStackTrace()) {
				s.print("\tat ");
				s.println(element);
			}
			int i = 1;
			for (Throwable ex : this.exceptions) {
				s.print("Nested exception ");
				s.print(i++);
				s.print(": ");
				ex.printStackTrace(s);
			}
		}
	}

	@Override
	public void printStackTrace(PrintWriter s) {
		synchronized (s) {
			s.println(this);
			for (StackTraceElement element : this.getStackTrace()) {
				s.print("\tat ");
				s.println(element);
			}
			int i = 1;
			for (Throwable ex : this.exceptions) {
				s.print("Nested exception ");
				s.print(i++);
				s.print(": ");
				ex.printStackTrace(s);
			}
		}
	}
}