package org.apache.maven.archetype.common;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import org.apache.maven.archetype.registry.ArchetypeRegistry;
import org.apache.maven.archetype.registry.io.xpp3.ArchetypeRegistryXpp3Reader;
import org.apache.maven.archetype.registry.io.xpp3.ArchetypeRegistryXpp3Writer;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.repository.ArtifactRepositoryFactory;
import org.apache.maven.artifact.repository.ArtifactRepositoryPolicy;
import org.apache.maven.artifact.repository.layout.ArtifactRepositoryLayout;
import org.codehaus.plexus.component.annotations.Component;
import org.codehaus.plexus.component.annotations.Requirement;
import org.codehaus.plexus.logging.AbstractLogEnabled;
import org.codehaus.plexus.util.IOUtil;
import org.codehaus.plexus.util.ReaderFactory;
import org.codehaus.plexus.util.StringUtils;
import org.codehaus.plexus.util.WriterFactory;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;

import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

@Component( role = ArchetypeRegistryManager.class )
public class DefaultArchetypeRegistryManager
    extends AbstractLogEnabled
    implements ArchetypeRegistryManager
{
    /**
     * Used to create ArtifactRepository objects given the urls of the remote repositories.
     */
    @Requirement
    private ArtifactRepositoryFactory artifactRepositoryFactory;

    /**
     * Determines whether the layout is legacy or not.
     */
    @Requirement
    private ArtifactRepositoryLayout defaultArtifactRepositoryLayout;

    public List<String> getFilteredExtensions( String archetypeFilteredExtentions, File archetypeRegistryFile )
        throws IOException
    {
        List<String> filteredExtensions = new ArrayList<String>();

        if ( StringUtils.isNotEmpty( archetypeFilteredExtentions ) )
        {
            filteredExtensions.addAll( Arrays.asList( StringUtils.split( archetypeFilteredExtentions, "," ) ) );
        }

        try
        {
            ArchetypeRegistry registry = readArchetypeRegistry( archetypeRegistryFile );

            filteredExtensions.addAll( registry.getFilteredExtensions() );
        }
        catch ( IOException e )
        {
            getLogger().warn( "Cannot read ~/.m2/archetype.xml" );
        }
        catch ( XmlPullParserException e )
        {
            getLogger().warn( "Cannot read ~/.m2/archetype.xml" );
        }

        if ( filteredExtensions.isEmpty() )
        {
            filteredExtensions.addAll( Constants.DEFAULT_FILTERED_EXTENSIONS );
        }

        return filteredExtensions;
    }

    public List<String> getLanguages( String archetypeLanguages, File archetypeRegistryFile )
        throws IOException
    {
        List<String> languages = new ArrayList<String>();

        if ( StringUtils.isNotEmpty( archetypeLanguages ) )
        {
            languages.addAll( Arrays.asList( StringUtils.split( archetypeLanguages, "," ) ) );
        }

        try
        {
            ArchetypeRegistry registry = readArchetypeRegistry( archetypeRegistryFile );

            languages.addAll( registry.getLanguages() );
        }
        catch ( IOException e )
        {
            getLogger().warn( "Can not read ~/.m2/archetype.xml" );
        }
        catch ( XmlPullParserException e )
        {
            getLogger().warn( "Can not read ~/.m2/archetype.xml" );
        }

        if ( languages.isEmpty() )
        {
            languages.addAll( Constants.DEFAULT_LANGUAGES );
        }

        return languages;
    }

    public ArchetypeRegistry readArchetypeRegistry( File archetypeRegistryFile )
        throws IOException, XmlPullParserException
    {
        if ( !archetypeRegistryFile.exists() )
        {
            return getDefaultArchetypeRegistry();
        }
        else
        {
            return readArchetypeRegistry( ReaderFactory.newXmlReader( archetypeRegistryFile ) );
        }
    }

    public ArchetypeRegistry readArchetypeRegistry( Reader reader )
        throws IOException, XmlPullParserException
    {
        ArchetypeRegistryXpp3Reader xpp3Reader = new ArchetypeRegistryXpp3Reader();

        try
        {
            return xpp3Reader.read( reader );
        }
        finally
        {
            IOUtil.close( reader );
        }
    }

    public void writeArchetypeRegistry( File archetypeRegistryFile, ArchetypeRegistry archetypeRegistry )
        throws IOException
    {
        ArchetypeRegistryXpp3Writer writer = new ArchetypeRegistryXpp3Writer();
        Writer out = WriterFactory.newXmlWriter( archetypeRegistryFile );

        try
        {
            writer.write( out, archetypeRegistry );
        }
        finally
        {
            IOUtil.close( out );
        }
    }

    /**
     * Code stealed from MavenArchetypeMojo
     * (org.apache.maven.plugins:maven-archetype-plugin:1.0-alpha4).
     */
    public ArtifactRepository createRepository( String url, String repositoryId )
    {
        // snapshots vs releases
        // offline = to turning the update policy off

        // TODO: we'll need to allow finer grained creation of repositories but this will do for now

        String updatePolicyFlag = ArtifactRepositoryPolicy.UPDATE_POLICY_ALWAYS;

        String checksumPolicyFlag = ArtifactRepositoryPolicy.CHECKSUM_POLICY_WARN;

        ArtifactRepositoryPolicy snapshotsPolicy =
            new ArtifactRepositoryPolicy( true, updatePolicyFlag, checksumPolicyFlag );

        ArtifactRepositoryPolicy releasesPolicy =
            new ArtifactRepositoryPolicy( true, updatePolicyFlag, checksumPolicyFlag );

        return artifactRepositoryFactory.createArtifactRepository( repositoryId, url, defaultArtifactRepositoryLayout,
                                                                   snapshotsPolicy, releasesPolicy );
    }

    public ArchetypeRegistry getDefaultArchetypeRegistry()
    {
        ArchetypeRegistry registry = new ArchetypeRegistry();

        registry.getLanguages().addAll( Constants.DEFAULT_LANGUAGES );

        registry.getFilteredExtensions().addAll( Constants.DEFAULT_FILTERED_EXTENSIONS );

        return registry;
    }
}
