"use strict";
/*---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.trimLastNewline = exports.LogOutputEvent = exports.logger = exports.Logger = exports.LogLevel = void 0;
const fs = require("fs");
const path = require("path");
const mkdirp = require("mkdirp");
const debugSession_1 = require("./debugSession");
var LogLevel;
(function (LogLevel) {
    LogLevel[LogLevel["Verbose"] = 0] = "Verbose";
    LogLevel[LogLevel["Log"] = 1] = "Log";
    LogLevel[LogLevel["Warn"] = 2] = "Warn";
    LogLevel[LogLevel["Error"] = 3] = "Error";
    LogLevel[LogLevel["Stop"] = 4] = "Stop";
})(LogLevel = exports.LogLevel || (exports.LogLevel = {}));
class Logger {
    constructor() {
        this._pendingLogQ = [];
    }
    log(msg, level = LogLevel.Log) {
        msg = msg + '\n';
        this._write(msg, level);
    }
    verbose(msg) {
        this.log(msg, LogLevel.Verbose);
    }
    warn(msg) {
        this.log(msg, LogLevel.Warn);
    }
    error(msg) {
        this.log(msg, LogLevel.Error);
    }
    dispose() {
        if (this._currentLogger) {
            const disposeP = this._currentLogger.dispose();
            this._currentLogger = null;
            return disposeP;
        }
        else {
            return Promise.resolve();
        }
    }
    /**
     * `log` adds a newline, `write` doesn't
     */
    _write(msg, level = LogLevel.Log) {
        // [null, undefined] => string
        msg = msg + '';
        if (this._pendingLogQ) {
            this._pendingLogQ.push({ msg, level });
        }
        else if (this._currentLogger) {
            this._currentLogger.log(msg, level);
        }
    }
    /**
     * Set the logger's minimum level to log in the console, and whether to log to the file. Log messages are queued before this is
     * called the first time, because minLogLevel defaults to Warn.
     */
    setup(consoleMinLogLevel, _logFilePath, prependTimestamp = true) {
        const logFilePath = typeof _logFilePath === 'string' ?
            _logFilePath :
            (_logFilePath && this._logFilePathFromInit);
        if (this._currentLogger) {
            const options = {
                consoleMinLogLevel,
                logFilePath,
                prependTimestamp
            };
            this._currentLogger.setup(options).then(() => {
                // Now that we have a minimum logLevel, we can clear out the queue of pending messages
                if (this._pendingLogQ) {
                    const logQ = this._pendingLogQ;
                    this._pendingLogQ = null;
                    logQ.forEach(item => this._write(item.msg, item.level));
                }
            });
        }
    }
    init(logCallback, logFilePath, logToConsole) {
        // Re-init, create new global Logger
        this._pendingLogQ = this._pendingLogQ || [];
        this._currentLogger = new InternalLogger(logCallback, logToConsole);
        this._logFilePathFromInit = logFilePath;
    }
}
exports.Logger = Logger;
exports.logger = new Logger();
/**
 * Manages logging, whether to console.log, file, or VS Code console.
 * Encapsulates the state specific to each logging session
 */
class InternalLogger {
    constructor(logCallback, isServer) {
        /** Dispose and allow exit to continue normally */
        this.beforeExitCallback = () => this.dispose();
        this._logCallback = logCallback;
        this._logToConsole = isServer;
        this._minLogLevel = LogLevel.Warn;
        this.disposeCallback = (signal, code) => {
            this.dispose();
            // Exit with 128 + value of the signal code.
            // https://nodejs.org/api/process.html#process_exit_codes
            code = code || 2; // SIGINT
            code += 128;
            process.exit(code);
        };
    }
    setup(options) {
        return __awaiter(this, void 0, void 0, function* () {
            this._minLogLevel = options.consoleMinLogLevel;
            this._prependTimestamp = options.prependTimestamp;
            // Open a log file in the specified location. Overwritten on each run.
            if (options.logFilePath) {
                if (!path.isAbsolute(options.logFilePath)) {
                    this.log(`logFilePath must be an absolute path: ${options.logFilePath}`, LogLevel.Error);
                }
                else {
                    const handleError = err => this.sendLog(`Error creating log file at path: ${options.logFilePath}. Error: ${err.toString()}\n`, LogLevel.Error);
                    try {
                        yield mkdirp(path.dirname(options.logFilePath));
                        this.log(`Verbose logs are written to:\n`, LogLevel.Warn);
                        this.log(options.logFilePath + '\n', LogLevel.Warn);
                        this._logFileStream = fs.createWriteStream(options.logFilePath);
                        this.logDateTime();
                        this.setupShutdownListeners();
                        this._logFileStream.on('error', err => {
                            handleError(err);
                        });
                    }
                    catch (err) {
                        handleError(err);
                    }
                }
            }
        });
    }
    logDateTime() {
        let d = new Date();
        let dateString = d.getUTCFullYear() + '-' + `${d.getUTCMonth() + 1}` + '-' + d.getUTCDate();
        const timeAndDateStamp = dateString + ', ' + getFormattedTimeString();
        this.log(timeAndDateStamp + '\n', LogLevel.Verbose, false);
    }
    setupShutdownListeners() {
        process.addListener('beforeExit', this.beforeExitCallback);
        process.addListener('SIGTERM', this.disposeCallback);
        process.addListener('SIGINT', this.disposeCallback);
    }
    removeShutdownListeners() {
        process.removeListener('beforeExit', this.beforeExitCallback);
        process.removeListener('SIGTERM', this.disposeCallback);
        process.removeListener('SIGINT', this.disposeCallback);
    }
    dispose() {
        return new Promise(resolve => {
            this.removeShutdownListeners();
            if (this._logFileStream) {
                this._logFileStream.end(resolve);
                this._logFileStream = null;
            }
            else {
                resolve();
            }
        });
    }
    log(msg, level, prependTimestamp = true) {
        if (this._minLogLevel === LogLevel.Stop) {
            return;
        }
        if (level >= this._minLogLevel) {
            this.sendLog(msg, level);
        }
        if (this._logToConsole) {
            const logFn = level === LogLevel.Error ? console.error :
                level === LogLevel.Warn ? console.warn :
                    null;
            if (logFn) {
                logFn(trimLastNewline(msg));
            }
        }
        // If an error, prepend with '[Error]'
        if (level === LogLevel.Error) {
            msg = `[${LogLevel[level]}] ${msg}`;
        }
        if (this._prependTimestamp && prependTimestamp) {
            msg = '[' + getFormattedTimeString() + '] ' + msg;
        }
        if (this._logFileStream) {
            this._logFileStream.write(msg);
        }
    }
    sendLog(msg, level) {
        // Truncate long messages, they can hang VS Code
        if (msg.length > 1500) {
            const endsInNewline = !!msg.match(/(\n|\r\n)$/);
            msg = msg.substr(0, 1500) + '[...]';
            if (endsInNewline) {
                msg = msg + '\n';
            }
        }
        if (this._logCallback) {
            const event = new LogOutputEvent(msg, level);
            this._logCallback(event);
        }
    }
}
class LogOutputEvent extends debugSession_1.OutputEvent {
    constructor(msg, level) {
        const category = level === LogLevel.Error ? 'stderr' :
            level === LogLevel.Warn ? 'console' :
                'stdout';
        super(msg, category);
    }
}
exports.LogOutputEvent = LogOutputEvent;
function trimLastNewline(str) {
    return str.replace(/(\n|\r\n)$/, '');
}
exports.trimLastNewline = trimLastNewline;
function getFormattedTimeString() {
    let d = new Date();
    let hourString = _padZeroes(2, String(d.getUTCHours()));
    let minuteString = _padZeroes(2, String(d.getUTCMinutes()));
    let secondString = _padZeroes(2, String(d.getUTCSeconds()));
    let millisecondString = _padZeroes(3, String(d.getUTCMilliseconds()));
    return hourString + ':' + minuteString + ':' + secondString + '.' + millisecondString + ' UTC';
}
function _padZeroes(minDesiredLength, numberToPad) {
    if (numberToPad.length >= minDesiredLength) {
        return numberToPad;
    }
    else {
        return String('0'.repeat(minDesiredLength) + numberToPad).slice(-minDesiredLength);
    }
}
//# sourceMappingURL=data:application/json;base64,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