/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.utils;

import static java.lang.String.valueOf;
import static org.apache.commons.lang3.StringUtils.defaultString;

import java.text.FieldPosition;
import java.text.Format;
import java.text.ParsePosition;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.commons.lang3.text.ExtendedMessageFormat;
import org.apache.commons.lang3.text.FormatFactory;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.LogMessages;
import org.eclipse.epp.logging.aeri.core.ILink;
import org.eclipse.epp.logging.aeri.core.ILinkable;
import org.eclipse.epp.logging.aeri.core.util.Logs;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;
import com.google.common.collect.Lists;

public class Formats {

    private static final CustomFormatFactory FACTORY = new CustomFormatFactory();

    private static final Map<String, CustomFormatFactory> REGISTRY = ImmutableMap.of(
            // @formatter:off
            "link", FACTORY, //$NON-NLS-1$
            "links", FACTORY, //$NON-NLS-1$
            "projects", FACTORY //$NON-NLS-1$
            // @formatter:on
    );

    private static final class CustomFormatFactory implements FormatFactory {

        @Override
        public Format getFormat(String name, final String arguments, Locale locale) {
            Builder<String, String> builder = ImmutableMap.builder();
            List<String> properties = Splitter.on(';').omitEmptyStrings().trimResults().splitToList(defaultString(arguments));
            for (String property : properties) {
                List<String> split = Splitter.on('=').omitEmptyStrings().trimResults().splitToList(property);
                builder.put(split.get(0), split.size() > 1 ? split.get(1) : valueOf(true));
            }
            ImmutableMap<String, String> args = builder.build();

            switch (name) {
            case "link": //$NON-NLS-1$
                return new LinkFormat(args);
            case "links": //$NON-NLS-1$
                return new LinkCollectionFormat(args);
            case "projects": //$NON-NLS-1$
                return new ProjectsFormat(args);
            default:
                return null;
            }
        }
    }

    private static class LinkCollectionFormat extends Format {

        private final Map<String, String> args;

        private LinkCollectionFormat(Map<String, String> args) {
            this.args = args;
        }

        @Override
        public StringBuffer format(Object obj, StringBuffer toAppendTo, FieldPosition pos) {
            Collection<ILink> list = null;
            Map<String, ILink> map = null;
            if (obj instanceof Collection) {
                list = (Collection) obj;
                Builder<String, ILink> b = ImmutableMap.builder();
                for (ILink link : list) {
                    b.put(link.getRel(), link);
                }
                map = b.build();
            } else if (obj instanceof ILink) {
                ILink link = (ILink) obj;
                map = ImmutableMap.of(link.getRel(), link);
            } else if (obj instanceof Map) {
                map = (Map) obj;
            } else if (obj instanceof ILinkable) {
                map = ((ILinkable) obj).getLinks().map();
            } else {
                return toAppendTo;
            }
            list = map.values();

            LinkFormat format = new LinkFormat(args);
            List<String> strings = Lists.newArrayList();
            for (ILink link : list) {
                String string = format.format(link);
                strings.add(string);
            }
            String separator = defaultString(args.get("separator"), " "); //$NON-NLS-1$ //$NON-NLS-2$
            String res = Joiner.on(separator).join(strings);
            return toAppendTo.append(res);
        }

        @Override
        public Object parseObject(String source, ParsePosition pos) {
            return null;
        }
    }

    private static class LinkFormat extends Format {
        private final Map<String, String> args;

        private LinkFormat(Map<String, String> args) {
            this.args = args;
        }

        @Override
        public StringBuffer format(Object obj, StringBuffer toAppendTo, FieldPosition pos) {

            ILink link = null;
            if (obj instanceof ILink) {
                link = (ILink) obj;
            } else if (obj instanceof Map) {
                String key = args.get("rel"); //$NON-NLS-1$
                link = (ILink) ((Map) obj).get(key);
            } else if (obj instanceof ILinkable) {
                String key = args.get("rel"); //$NON-NLS-1$
                link = ((ILinkable) obj).getLinks().get(key);
            } else {
                return toAppendTo;
            }
            if (link == null) {
                return toAppendTo.append("null"); //$NON-NLS-1$
            }
            if (useBrackets()) {
                toAppendTo.append('[');
            }
            String title = defaultString(args.get("title"), link.getTitle()); //$NON-NLS-1$
            toAppendTo.append("<a href=\"").append(link.getHref()).append("\">").append(title).append("</a>"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
            if (useBrackets()) {
                toAppendTo.append(']');
            }
            return toAppendTo;
        }

        private boolean useBrackets() {
            return "true".equals(args.get("brackets")); //$NON-NLS-1$ //$NON-NLS-2$
        }

        @Override
        public Object parseObject(String source, ParsePosition pos) {
            return null;
        }
    }

    public static String format(String pattern, Object... args) {
        // we tolerate null but do not do anything meaningful...
        pattern = defaultString(pattern, "null"); //$NON-NLS-1$
        try {
            return new ExtendedMessageFormat(pattern, REGISTRY).format(args);
        } catch (Exception e) {
            Logs.log(LogMessages.WARN_FORMATTING_FAILED, e, pattern, args, e.getMessage());
        }
        return pattern;
    }

}
