/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.logging.aeri.core.filters;

import static org.eclipse.epp.logging.aeri.core.filters.RequiredPackagesFilter.TestResult.UNDEF;

import java.util.List;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.epp.logging.aeri.core.util.StatusSwitch;
import org.eclipse.jdt.annotation.Nullable;

import com.google.common.base.MoreObjects;
import com.google.common.base.Predicate;

public class RequiredPackagesFilter implements Predicate<IStatus> {

    private List<Pattern> patterns;

    /**
     * @param patterns
     *            the list of package patterns to match
     */
    public RequiredPackagesFilter(List<Pattern> patterns) {
        this.patterns = patterns;
    }

    @Override
    public boolean apply(IStatus input) {
        // if nothing is required - all are welcome...
        if (patterns.isEmpty()) {
            return true;
        }
        RequiredPackageSwitch test = new RequiredPackageSwitch();
        TestResult res = MoreObjects.firstNonNull(test.doSwitch(input), UNDEF);
        switch (res) {
        case PASS:
            return true;
        case FAIL:
        case UNDEF:
        default:
            return false;
        }
    }

    private final class RequiredPackageSwitch extends StatusSwitch<TestResult> {

        @Override
        @Nullable
        public TestResult caseStackTraceElement(StackTraceElement stackTraceElement) {
            String label = stackTraceElement.getClassName() + "." + stackTraceElement.getMethodName();
            for (Pattern pattern : patterns) {
                if (pattern.matcher(label).matches()) {
                    return TestResult.PASS;
                }
            }
            return null;
        }
    }

    enum TestResult {
        PASS, FAIL, UNDEF
    }
}
