/*******************************************************************************
 * Copyright (c) 2010-2016, Abel Hegedus, IncQuery Labs Ltd.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 *******************************************************************************/
package org.eclipse.viatra.query.runtime.registry;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.viatra.query.runtime.IExtensions;
import org.eclipse.viatra.query.runtime.api.IQueryGroup;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.extensibility.IQueryGroupProvider;
import org.eclipse.viatra.query.runtime.extensibility.IQuerySpecificationProvider;
import org.eclipse.viatra.query.runtime.util.ViatraQueryLoggingUtil;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;

/**
 * Loader for the {@link QuerySpecificationRegistry} based on the query group extensions generated by the VIATRA Query
 * builder. The loader has a single instance that processes the extensions on demand if the platform is running, caches
 * the results and updates the {@link QuerySpecificationRegistry}. Note that the loader does not perform class loading
 * on the query group if possible.
 * 
 * <p>
 * The class has a single instance accessible with {@link #getInstance()}.
 * 
 * @author Abel Hegedus
 * @since 1.3
 *
 */
public class ExtensionBasedQuerySpecificationLoader {

    public static final String CONNECTOR_ID = "org.eclipse.viatra.query.runtime.querygroup.extension.based.connector";

    private static final String DUPLICATE_QUERY_GROUP_MESSAGE = "Duplicate query group identifier %s for plugin %s (already contributed by %s)";
    private static final ExtensionBasedQuerySpecificationLoader INSTANCE = new ExtensionBasedQuerySpecificationLoader();
    
    private Multimap<String, String>  contributingPluginOfGroupMap = HashMultimap.create();
    private Map<String, QueryGroupProvider> contributedQueryGroups;

    private ExtensionBasedSourceConnector sourceConnector;

    
    /**
     * @return the single instance of the loader.
     */
    public static ExtensionBasedQuerySpecificationLoader getInstance() {
        return INSTANCE;
    }
    
    /**
     * Loads the query specifications that are registered through extension points into the
     * {@link QuerySpecificationRegistry}.
     */
    public void loadRegisteredQuerySpecificationsIntoRegistry() {
        ((QuerySpecificationRegistry) QuerySpecificationRegistry.getInstance()).addDelayedSourceConnector(getSourceConnector());
    }
    
    /**
     * Return a source connector that can be used to load query specifications contributed through
     * extensions into a {@link IQuerySpecificationRegistry}.
     * 
     * @return the source connector
     */
    public IRegistrySourceConnector getSourceConnector() {
        if (this.sourceConnector == null) {
            this.sourceConnector = new ExtensionBasedSourceConnector();
        }
        return sourceConnector; 
    }

    private Map<String, QueryGroupProvider> getRegisteredQueryGroups() {
        if(contributedQueryGroups != null) {
            return contributedQueryGroups;
        }
        contributedQueryGroups = new HashMap<>();
        if (Platform.isRunning()) {
            for (IConfigurationElement e : Platform.getExtensionRegistry().getConfigurationElementsFor(IExtensions.QUERY_SPECIFICATION_EXTENSION_POINT_ID)) {
                if (e.isValid()) {
                    processExtension(e);
                }
            }
        }
        return contributedQueryGroups;
    }
    
    private void processExtension(IConfigurationElement el) {
        String id = null;
        try {
            String contributorName = el.getContributor().getName();
            id = el.getAttribute("id");
            if(id == null) {
                throw new IllegalStateException(String.format("Query group extension identifier is required (plug-in: %s)!", contributorName));
            }
            
            QueryGroupProvider provider = new QueryGroupProvider(el);
            
            QueryGroupProvider queryGroupInMap = contributedQueryGroups.get(id);
            if(queryGroupInMap != null) {
                Collection<String> contributorPlugins = contributingPluginOfGroupMap.get(id);
                throw new IllegalStateException(String.format(DUPLICATE_QUERY_GROUP_MESSAGE, id, contributorName, contributorPlugins));
            }
            
            contributedQueryGroups.put(id, provider);
            contributingPluginOfGroupMap.put(id, contributorName);
        } catch (Exception e) {
            // If there are serious compilation errors in the file loaded by the query registry, an error is thrown
            if (id == null) {
                id = "undefined in plugin.xml";
            }
            ViatraQueryLoggingUtil.getLogger(ExtensionBasedQuerySpecificationLoader.class).error(
                    "[ExtensionBasedQuerySpecificationLoader] Exception during query specification registry initialization when preparing group: "
                            + id + "! " + e.getMessage(), e);
        }
    }

    /**
     * @author Abel Hegedus
     *
     */
    private final class ExtensionBasedSourceConnector implements IRegistrySourceConnector {
        
        private Set<IConnectorListener> listeners;
        
        public ExtensionBasedSourceConnector() {
            this.listeners = new HashSet<>();
        }
        
        @Override
        public String getIdentifier() {
            return ExtensionBasedQuerySpecificationLoader.CONNECTOR_ID;
        }

        @Override
        public void addListener(IConnectorListener listener) {
            Objects.requireNonNull(listener, "Listener must not be null!");
            boolean added = listeners.add(listener);
            if(added) {
                for (QueryGroupProvider queryGroupProvider : getRegisteredQueryGroups().values()) {
                    for (IQuerySpecificationProvider specificationProvider : queryGroupProvider.getQuerySpecificationProviders()) {
                        listener.querySpecificationAdded(this, specificationProvider);
                    }
                }
            }
        }

        @Override
        public void removeListener(IConnectorListener listener) {
            Objects.requireNonNull(listener, "Listener must not be null!");
            listeners.remove(listener);
        }

        @Override
        public boolean includeSpecificationsInDefaultViews() {
            return true;
        }
    }

    /**
     * Provider implementation that uses the group extension to load the query group on-demand.
     * It also provides the set of query FQNs that are part of the group without class loading.
     * Once loaded, the query group is cached for future use.
     * 
     * @author Abel Hegedus
     */
    private static final class QueryGroupProvider implements IQueryGroupProvider {
    
        private static final String DUPLICATE_FQN_MESSAGE = "Duplicate FQN %s in query group extension point (plug-in %s)";
        private final IConfigurationElement element;
        private IQueryGroup queryGroup;
        private Set<String> querySpecificationFQNs;
        private Map<String, IQuerySpecificationProvider> querySpecificationMap;
        
        public QueryGroupProvider(IConfigurationElement element) {
            this.element = element;
            this.queryGroup = null;
            this.querySpecificationFQNs = null;
            this.querySpecificationMap = null;
        }
        
        @Override
        public IQueryGroup get() {
            try{
                if(queryGroup == null) {
                    queryGroup = (IQueryGroup) element.createExecutableExtension("group");
                }
                return queryGroup;
            } catch (CoreException e) {
                throw new IllegalStateException(e.getMessage(), e);
            }
        }
        
        @Override
        public Set<String> getQuerySpecificationFQNs() {
            if(querySpecificationFQNs == null) {
                Set<String> fqns = new HashSet<>();
                for (IConfigurationElement e : element.getChildren("query-specification")) {
                    if (e.isValid()) {
                        String fqn = e.getAttribute("fqn");
                        boolean added = fqns.add(fqn);
                        if(!added) {
                            String contributorName = e.getContributor().getName();
                            throw new IllegalArgumentException(String.format(DUPLICATE_FQN_MESSAGE,fqn, contributorName));
                        }
                    }
                }
                if(fqns.isEmpty()) {
                    // we must load the class and get the specifications
                    IQueryGroup loadedQueryGroup = get();
                    for (IQuerySpecification<?> specification : loadedQueryGroup.getSpecifications()) {
                        String fullyQualifiedName = specification.getFullyQualifiedName();
                        boolean added = fqns.add(fullyQualifiedName);
                        if(!added) {
                            String contributorName = element.getContributor().getName();
                            throw new IllegalArgumentException(String.format(DUPLICATE_FQN_MESSAGE, fullyQualifiedName, contributorName));
                        }
                    }
                }
                // we will never change the set after initialization
                querySpecificationFQNs = new HashSet<>(fqns);
            }
            return querySpecificationFQNs;
        }
        
        @Override
        public Set<IQuerySpecificationProvider> getQuerySpecificationProviders() {
            return new HashSet<>(getQuerySpecificationMap().values());
        }

        private Map<String, IQuerySpecificationProvider> getQuerySpecificationMap() {
            if(querySpecificationMap == null){
                querySpecificationMap = new HashMap<>();
                Set<String> fqns = getQuerySpecificationFQNs();
                for (String fqn : fqns) {
                    querySpecificationMap.put(fqn, new GroupBasedQuerySpecificationProvider(fqn, this));
                }
            }
            return querySpecificationMap;
        }

    }
    
    /**
     * Provider implementation that uses the query group extension to load a query specification by its FQN. Note that
     * the FQN of the provided query specification is set with the constructor and can be requested without loading the
     * class. Once loaded, the query specification is cached for future use.
     * 
     * @author Abel Hegedus
     *
     */
    private static final class GroupBasedQuerySpecificationProvider implements IQuerySpecificationProvider {

        private String queryFQN;
        private QueryGroupProvider queryGroupProvider;
        private IQuerySpecification<?> specification;

        public GroupBasedQuerySpecificationProvider(String queryFQN, QueryGroupProvider queryGroupProvider) {
            this.queryFQN = queryFQN;
            this.queryGroupProvider = queryGroupProvider;
            this.specification = null;
        }
        
        @Override
        public IQuerySpecification<?> get() {
            if(specification == null) {
                if(queryGroupProvider.getQuerySpecificationFQNs().contains(queryFQN)) {
                    for (IQuerySpecification<?> spec : queryGroupProvider.get().getSpecifications()) {
                        if(spec.getFullyQualifiedName().equals(queryFQN)){
                            this.specification = spec;
                        }
                    }
                } else {
                    throw new IllegalStateException(String.format("Could not find query specifition %s in group (plug-in %s)", queryFQN, queryGroupProvider.element.getContributor().getName()));
                }
            }
            return specification;
        }

        @Override
        public String getFullyQualifiedName() {
            return queryFQN;
        }
        
        @Override
        public String getSourceProjectName() {
            return queryGroupProvider.element.getContributor().getName();
        }
    }
}
