/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.logging.aeri.core.filters;

import static org.apache.commons.lang3.StringUtils.defaultString;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.epp.logging.aeri.core.util.StatusSwitch;

import com.google.common.base.Predicate;

public class StatusSizeLimitFilter implements Predicate<IStatus> {

    private int maxSizeInBytes;

    public StatusSizeLimitFilter(int maxSizeInBytes) {
        this.maxSizeInBytes = maxSizeInBytes;
    }

    @Override
    public boolean apply(IStatus report) {
        // in case of very large reports (~50+ MB), toJson or String.getBytes may be too expensive
        StatusSizeLimitFilter.EstimateSizeVisitor visitor = new EstimateSizeVisitor();
        visitor.doSwitch(report);
        // assume 16 bit chars
        int estimatedByteLength = visitor.estimatedCharacters * 2;
        return estimatedByteLength < maxSizeInBytes;
    }

    private static final class EstimateSizeVisitor extends StatusSwitch<Object> {

        int estimatedCharacters = 0;

        @Override
        public Object caseStatus(IStatus status) {
            estimatedCharacters += defaultString(status.getMessage()).length();
            return null;
        }

        @Override
        public Object caseThrowable(Throwable throwable) {
            estimatedCharacters += defaultString(throwable.getMessage()).length();
            return null;
        }

        @Override
        public Object caseStackTraceElement(StackTraceElement element) {
            estimatedCharacters += defaultString(element.getClassName()).length();
            estimatedCharacters += defaultString(element.getFileName()).length();
            estimatedCharacters += defaultString(element.getMethodName()).length();
            return null;
        }
    }
}
