/**
 * Copyright (c) 2006-2020 Borland Software Corporation, CEA LIST, Artal and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *    Dmitry Stadnik (Borland) - initial API and implementation
 *    Michael Golubev (Montages) - #386838 - migrate to Xtend2
 *    Aurelien Didier (ARTAL) - aurelien.didier51@gmail.com - Bug 569174
 */
package xpt;

import com.google.common.base.Objects;
import com.google.inject.Singleton;
import java.util.Set;
import org.eclipse.emf.codegen.ecore.genmodel.GenClass;
import org.eclipse.emf.codegen.ecore.genmodel.GenClassifier;
import org.eclipse.emf.codegen.ecore.genmodel.GenFeature;
import org.eclipse.emf.codegen.ecore.genmodel.GenJDKLevel;
import org.eclipse.emf.codegen.ecore.genmodel.impl.GenFeatureImpl;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;

@Singleton
@SuppressWarnings("all")
public class GenModelUtils_qvto {
  private static final Set<String> PRIMITIVES = CollectionLiterals.<String>newHashSet("boolean", "byte", "char", "double", "float", "int", "long", 
    "short");
  
  /**
   * GenModel functionality unrelated to EMF generated code
   * or metamodel access only.
   */
  public GenFeature getReverse(final GenFeature feature) {
    return feature.getReverse();
  }
  
  /**
   * For primitive types, return their java.lang wrappers, for non-primitive
   * types returns instanceClassName as is
   */
  public String getObjectCompatibleClassName(final EDataType dt) {
    final String instanceClass = dt.getInstanceClassName();
    boolean _contains = instanceClass.contains(".");
    if (_contains) {
      return instanceClass;
    }
    String _switchResult = null;
    boolean _matched = false;
    if (Objects.equal(instanceClass, "boolean")) {
      _matched=true;
      return "java.lang.Boolean";
    }
    if (!_matched) {
      if (Objects.equal(instanceClass, "byte")) {
        _matched=true;
        return "java.lang.Byte";
      }
    }
    if (!_matched) {
      if (Objects.equal(instanceClass, "char")) {
        _matched=true;
        return "java.lang.Character";
      }
    }
    if (!_matched) {
      if (Objects.equal(instanceClass, "double")) {
        _matched=true;
        return "java.lang.Double";
      }
    }
    if (!_matched) {
      if (Objects.equal(instanceClass, "float")) {
        _matched=true;
        return "java.lang.Float";
      }
    }
    if (!_matched) {
      if (Objects.equal(instanceClass, "int")) {
        _matched=true;
        return "java.lang.Integer";
      }
    }
    if (!_matched) {
      if (Objects.equal(instanceClass, "long")) {
        _matched=true;
        return "java.lang.Long";
      }
    }
    if (!_matched) {
      if (Objects.equal(instanceClass, "short")) {
        _matched=true;
        return "java.lang.Short";
      }
    }
    if (!_matched) {
      _switchResult = "java.lang.Object";
    }
    return _switchResult;
  }
  
  public GenClassifier getTypeGenClassifier(final GenFeature feature) {
    return feature.getTypeGenClassifier();
  }
  
  /**
   * FIXME: for models loaded from generated package (e.g. gmf tests often
   * refer to EClass/EOperation directly from EcorePackage) instanceClassName
   * is always set and thus all metamodel types are treated as external interfaces,
   * with useless casts to EObject
   * 
   * GenClass#isExternalInterface()
   */
  public boolean isExternalInterface(final GenClass gc) {
    return gc.isExternalInterface();
  }
  
  public boolean isContains(final GenFeature gf) {
    return ((GenFeatureImpl) gf).isContains();
  }
  
  public boolean isPrimitiveType(final GenFeature gf) {
    return GenModelUtils_qvto.PRIMITIVES.contains(gf.getEcoreFeature().getEType().getInstanceClassName());
  }
  
  public boolean isSuperTypeOf(final GenClass xptSelf, final GenClass subclassCandidate) {
    if ((Objects.equal(xptSelf.getEcoreClass().getName(), "EObject") && 
      Objects.equal(xptSelf.getEcoreClass().getEPackage().getNsURI(), "http://www.eclipse.org/emf/2002/Ecore"))) {
      return true;
    }
    return xptSelf.getEcoreClass().isSuperTypeOf(subclassCandidate.getEcoreClass());
  }
  
  /**
   * @returns 4 for Java1.4, 5 for JDK 5.0, 6 for JDK6.0 and 7 for JDK7.0
   */
  public int jdkComplianceLevel(final GenClass xptSelf) {
    int _xblockexpression = (int) 0;
    {
      GenJDKLevel _xifexpression = null;
      boolean _equals = Objects.equal(xptSelf, null);
      if (_equals) {
        _xifexpression = GenJDKLevel.JDK60_LITERAL;
      } else {
        _xifexpression = xptSelf.getGenPackage().getGenModel().getComplianceLevel();
      }
      GenJDKLevel l = _xifexpression;
      int _switchResult = (int) 0;
      if (l != null) {
        switch (l) {
          case JDK14_LITERAL:
            _switchResult = 4;
            break;
          case JDK50_LITERAL:
            _switchResult = 5;
            break;
          case JDK60_LITERAL:
            _switchResult = 6;
            break;
          case JDK70_LITERAL:
            _switchResult = 7;
            break;
          default:
            _switchResult = 6;
            break;
        }
      } else {
        _switchResult = 6;
      }
      _xblockexpression = _switchResult;
    }
    return _xblockexpression;
  }
}
