/**
 * Copyright (c) 2014, 2015 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Christian W. Damus - Initial API and implementation
 */
package org.eclipse.papyrus.uml.profile.assistants.generator;

import java.util.ArrayList;
import java.util.HashMap;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.eclipse.papyrus.infra.filters.Filter;
import org.eclipse.papyrus.infra.gmfdiag.assistant.AssistantFactory;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ElementTypeFilter;
import org.eclipse.papyrus.infra.gmfdiag.assistant.PopupAssistant;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.uml.profile.assistants.generator.FiltersUtil;
import org.eclipse.papyrus.uml.profile.types.generator.Identifiers;
import org.eclipse.papyrus.uml.profile.types.generator.ImpliedExtension;
import org.eclipse.papyrus.uml.profile.types.generator.UML;
import org.eclipse.papyrus.uml.profile.types.generator.UMLElementTypes;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Transformation rule for generating a {@link PopupAssistant} from a UML {@link Extension}.
 */
@Singleton
@SuppressWarnings("all")
public class PopupAssistantRule {
  @Extension
  private static AssistantFactory assistantFactory = AssistantFactory.eINSTANCE;
  
  @Inject
  @Extension
  private Identifiers _identifiers;
  
  @Inject
  @Extension
  private UMLElementTypes _uMLElementTypes;
  
  @Inject
  @Extension
  private FiltersUtil _filtersUtil;
  
  @Inject
  @Extension
  private UML _uML;
  
  public PopupAssistant toPopupAssistant(final ImpliedExtension umlExtension, final ElementTypeConfiguration basetype) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlExtension, basetype);
    final PopupAssistant _result;
    synchronized (_createCache_toPopupAssistant) {
      if (_createCache_toPopupAssistant.containsKey(_cacheKey)) {
        return _createCache_toPopupAssistant.get(_cacheKey);
      }
      PopupAssistant _createPopupAssistant = PopupAssistantRule.assistantFactory.createPopupAssistant();
      _result = _createPopupAssistant;
      _createCache_toPopupAssistant.put(_cacheKey, _result);
    }
    _init_toPopupAssistant(_result, umlExtension, basetype);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, PopupAssistant> _createCache_toPopupAssistant = CollectionLiterals.newHashMap();
  
  private void _init_toPopupAssistant(final PopupAssistant it, final ImpliedExtension umlExtension, final ElementTypeConfiguration basetype) {
    it.setElementTypeID(this._identifiers.toElementTypeID(umlExtension, basetype));
    it.setOwnedFilter(this._filtersUtil.andProfileApplied(this._filtersUtil.reduce(this.createPossibleOwnersFilter(basetype, umlExtension)), this._uML.getProfile(umlExtension)));
  }
  
  private Filter createPossibleOwnersFilter(final ElementTypeConfiguration basetype, final ImpliedExtension umlExtension) {
    final Function2<Filter, ElementTypeConfiguration, Filter> _function = new Function2<Filter, ElementTypeConfiguration, Filter>() {
      @Override
      public Filter apply(final Filter filter, final ElementTypeConfiguration elementType) {
        Filter _xifexpression = null;
        boolean _canContainType = PopupAssistantRule.this._uMLElementTypes.canContainType(elementType, basetype);
        if (_canContainType) {
          ElementTypeFilter _filter = PopupAssistantRule.this._filtersUtil.toFilter(elementType, PopupAssistantRule.this._uML.getProfile(umlExtension));
          _xifexpression = PopupAssistantRule.this._filtersUtil.operator_or(filter, _filter);
        } else {
          _xifexpression = filter;
        }
        return _xifexpression;
      }
    };
    return IterableExtensions.<ElementTypeConfiguration, Filter>fold(this._uMLElementTypes.getBaseElementTypes(), null, _function);
  }
}
