/*****************************************************************************
 * Copyright (c) 2023, 2024 CEA LIST, Obeo.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Obeo - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrus.sirius.junit.utils.diagram.creation.graphical.checker;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmf.runtime.notation.Diagram;
import org.eclipse.osgi.util.NLS;
import org.eclipse.sirius.diagram.DEdge;
import org.eclipse.sirius.viewpoint.DRepresentationElement;
import org.junit.Assert;

/**
 * Abstract GraphicalEdgeChecker for creation tests.
 *
 * @author <a href="mailto:jessy.mallet@obeo.fr">Jessy Mallet</a>
 */
public abstract class AbstractGraphicalEdgeCreationChecker extends AbstractGraphicalCreationChecker {

	/**
	 * The initial number of edge on the Sirius Diagram.
	 */
	protected final int nbGraphicalEdge;

	/**
	 * The expected number of additional edges.
	 */
	protected int expectedAdditionalEdges = DEFAULT_CREATED_ELEMENTS_NUMBER;


	/**
	 * Constructor.
	 *
	 * @param diagram
	 *            the gmf diagram
	 * @param graphicalParent
	 *            the graphical parent
	 */
	public AbstractGraphicalEdgeCreationChecker(final Diagram diagram, final EObject graphicalParent) {
		super(diagram, graphicalParent);
		nbGraphicalEdge = diagram.getEdges().size();
	}


	/**
	 * Check the type of the created element
	 *
	 * @param createdElementRepresentation
	 *            the created {@link DRepresentationElement}
	 */
	@Override
	protected void checkCreatedElementInstanceOf(final DRepresentationElement createdView) {
		Assert.assertTrue(NLS.bind("The created element must be a DEdge instead of a {0}.", createdView.eClass().getName()),  //$NON-NLS-1$
				createdView instanceof DEdge);
	}

	/**
	 * Check the mapping of the created element
	 *
	 * @param createdElementRepresentation
	 *            the created {@link DRepresentationElement}
	 */
	@Override
	protected void checkCreatedElementMapping(final DRepresentationElement createdView) {
		Assert.assertEquals("The mapping is not the expected one.", getEdgeMappingType(), createdView.getMapping().getName()); //$NON-NLS-1$

		// check the parent of the created element is the expected one.
		Assert.assertEquals("Parent of the created graphical element is wrong", //$NON-NLS-1$
				this.graphicalParent, createdView.eContainer());
	}

	@Override
	protected void checkElementsSizes(String phase, boolean done) {
		super.checkElementsSizes(phase, done);

		int newEdges = getExpectedAdditionalEdges();
		if (!done) {
			newEdges = 0;
		}

		Assert.assertEquals("Unexpected number of additional edge on " + phase, //$NON-NLS-1$
				newEdges, diagram.getEdges().size() - nbGraphicalEdge);
	}


	/**
	 * Get the expected number of additional edges.
	 *
	 * @return the expected number of additional edges.
	 */
	public int getExpectedAdditionalEdges() {
		return this.expectedAdditionalEdges;
	}

	/**
	 * Set the expected number of additional edges.
	 *
	 * @param expectedEdges
	 *            the expected number of additional edges.
	 */
	public void setExpectedAdditionalEdges(int expectedEdges) {
		this.expectedAdditionalEdges = expectedEdges;
	}

	/**
	 *
	 * @return
	 *         the expected mapping type for the created edge
	 */
	protected abstract String getEdgeMappingType();
}