/**
 * Copyright (c) 2019 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.ros2.codegen.cpp.utils;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.eclipse.cdt.core.CCProjectNature;
import org.eclipse.cdt.core.CCorePlugin;
import org.eclipse.cdt.core.dom.IPDOMManager;
import org.eclipse.cdt.core.envvar.EnvironmentVariable;
import org.eclipse.cdt.core.envvar.IContributedEnvironment;
import org.eclipse.cdt.core.envvar.IEnvironmentVariable;
import org.eclipse.cdt.core.formatter.DefaultCodeFormatterConstants;
import org.eclipse.cdt.core.index.IIndexManager;
import org.eclipse.cdt.core.model.CoreModel;
import org.eclipse.cdt.core.settings.model.CIncludePathEntry;
import org.eclipse.cdt.core.settings.model.ICConfigurationDescription;
import org.eclipse.cdt.core.settings.model.ICFolderDescription;
import org.eclipse.cdt.core.settings.model.ICLanguageSetting;
import org.eclipse.cdt.core.settings.model.ICLanguageSettingEntry;
import org.eclipse.cdt.core.settings.model.ICProjectDescription;
import org.eclipse.cdt.core.settings.model.ICProjectDescriptionManager;
import org.eclipse.cdt.core.settings.model.ICSettingEntry;
import org.eclipse.cdt.managedbuilder.core.IBuilder;
import org.eclipse.cdt.managedbuilder.core.IConfiguration;
import org.eclipse.cdt.managedbuilder.core.ManagedBuildManager;
import org.eclipse.cdt.newmake.core.IMakeBuilderInfo;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.papyrus.designer.languages.common.extensionpoints.ILangProjectSupport;
import org.eclipse.papyrus.designer.languages.common.extensionpoints.LanguageProjectSupport;
import org.eclipse.papyrus.designer.languages.cpp.codegen.utils.CppClassUtils;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.External;
import org.eclipse.papyrus.designer.transformation.base.utils.ProjectManagement;
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException;
import org.eclipse.papyrus.designer.transformation.core.transformations.ExecuteTransformationChain;
import org.eclipse.papyrus.designer.transformation.core.transformations.TransformationContext;
import org.eclipse.papyrus.robotics.ros2.base.EnvironmentUtils;
import org.eclipse.papyrus.robotics.ros2.base.Ros2Constants;
import org.eclipse.papyrus.robotics.ros2.codegen.common.RoboticsTContext;
import org.eclipse.papyrus.robotics.ros2.codegen.cpp.component.RoboticsCppCreator;
import org.eclipse.papyrus.robotics.ros2.preferences.Ros2Distributions;
import org.eclipse.papyrus.robotics.ros2.preferences.Ros2PreferenceUtils;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Exceptions;

/**
 * get or create a CDT project with a given name
 */
@SuppressWarnings("all")
public class ProjectTools {
  /**
   * get a CDT project with a given name. If the CDT project does not
   * exist, it will be created. If the project exists, but is not a C++
   * CDT project, it will be converted to CDT.
   * The project will be configured for a colcon build.
   */
  public static IProject getProject(final String projectName) {
    try {
      IProject genProject = ProjectManagement.getNamedProject(projectName);
      if (((((genProject != null) && genProject.exists()) && 
        (genProject.getNature(CCProjectNature.CC_NATURE_ID) == null)) || 
        RoboticsTContext.rewriteProject(projectName))) {
        genProject = null;
      }
      if (((genProject == null) || (!genProject.exists()))) {
        final ILangProjectSupport projectSupport = LanguageProjectSupport.getProjectSupport("C++");
        final String currentIndexer = CCorePlugin.getIndexManager().getDefaultIndexerId();
        IIndexManager _indexManager = CCorePlugin.getIndexManager();
        _indexManager.setDefaultIndexerId(IPDOMManager.ID_NO_INDEXER);
        genProject = projectSupport.createProject(projectName, TransformationContext.current.modelRoot);
        IIndexManager _indexManager_1 = CCorePlugin.getIndexManager();
        _indexManager_1.setDefaultIndexerId(currentIndexer);
        if (((genProject != null) && (!genProject.exists()))) {
          String _format = String.format(
            "project does not exist");
          throw new RuntimeException(_format);
        }
        if ((genProject != null)) {
          ProjectTools.configureCDT(genProject, projectName.toLowerCase());
        } else {
          throw new TransformationException(ExecuteTransformationChain.USER_CANCEL);
        }
      }
      return genProject;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Basic configuration of a CDT project for colcon
   */
  public static void configureCDT(final IProject project, final String pkgName) {
    try {
      final ICProjectDescriptionManager mngr = CoreModel.getDefault().getProjectDescriptionManager();
      ICProjectDescription cdesc = mngr.getProjectDescription(project, true);
      final ProjectScope scope = new ProjectScope(project);
      final IEclipsePreferences ref = scope.getNode(CCorePlugin.PLUGIN_ID);
      ref.put(DefaultCodeFormatterConstants.FORMATTER_TAB_CHAR, "space");
      ref.put(DefaultCodeFormatterConstants.FORMATTER_TAB_SIZE, "2");
      try {
        final String amentPrefix = EnvironmentUtils.get(Ros2Constants.AMENT_PREFIX_PATH);
        final String cmakePrefix = EnvironmentUtils.get(Ros2Constants.CMAKE_PREFIX_PATH);
        final String pythonPath = EnvironmentUtils.get(Ros2Constants.PYTHON_PATH);
        ICConfigurationDescription[] _configurations = cdesc.getConfigurations();
        for (final ICConfigurationDescription configDescr : _configurations) {
          {
            ProjectTools.addEnvironmenVar(configDescr, Ros2Constants.AMENT_PREFIX_PATH, amentPrefix);
            ProjectTools.addEnvironmenVar(configDescr, Ros2Constants.CMAKE_PREFIX_PATH, cmakePrefix);
            ProjectTools.addEnvironmenVar(configDescr, Ros2Constants.PYTHON_PATH, pythonPath);
            final IConfiguration main = ManagedBuildManager.getConfigurationForDescription(configDescr);
            main.setBuildCommand("colcon");
            final IBuilder builder = main.getBuilder();
            builder.setUseDefaultBuildCmdOnly(false);
            builder.setBuildPath("${workspace_loc:}");
            String buildCmd = String.format(
              "build %s %s %s", 
              Ros2PreferenceUtils.getColconOptions(), 
              Ros2PreferenceUtils.getColconPackageOptions(), pkgName);
            String _name = configDescr.getName();
            boolean _equals = Objects.equals(_name, "Debug");
            if (_equals) {
              String _buildCmd = buildCmd;
              buildCmd = (_buildCmd + " --cmake-args  -DCMAKE_BUILD_TYPE=Debug");
            }
            builder.setBuildAttribute(IMakeBuilderInfo.BUILD_TARGET_INCREMENTAL, buildCmd);
            main.setManagedBuildOn(false);
          }
        }
        mngr.setProjectDescription(project, cdesc, true, null);
        ManagedBuildManager.saveBuildInfo(project, true);
        TransformationContext.monitor.subTask("waiting for CDT to finish project setup");
        ProjectTools.waitForCDT();
      } catch (final Throwable _t) {
        if (_t instanceof CoreException) {
          final CoreException ce = (CoreException)_t;
          String _message = ce.getMessage();
          throw new RuntimeException(_message);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Add a environment variable, if non null
   */
  public static IEnvironmentVariable addEnvironmenVar(final ICConfigurationDescription configDescr, final String key, final String value) {
    IEnvironmentVariable _xifexpression = null;
    if ((value != null)) {
      IEnvironmentVariable _xblockexpression = null;
      {
        final IContributedEnvironment contribEnv = CCorePlugin.getDefault().getBuildEnvironmentManager().getContributedEnvironment();
        EnvironmentVariable _environmentVariable = new EnvironmentVariable(key, value);
        _xblockexpression = contribEnv.addVariable(_environmentVariable, configDescr);
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }

  /**
   * Wait up to 10 seconds for the CDT indexer to finish
   */
  public static void waitForCDT() {
    boolean _isIndexerIdle = CCorePlugin.getIndexManager().isIndexerIdle();
    boolean _not = (!_isIndexerIdle);
    if (_not) {
      TransformationContext.monitor.subTask("waiting for CDT indexer");
      int i = 0;
      do {
        {
          try {
            Thread.sleep(100);
          } catch (final Throwable _t) {
            if (_t instanceof InterruptedException) {
            } else {
              throw Exceptions.sneakyThrow(_t);
            }
          }
          i++;
        }
      } while(((!CCorePlugin.getIndexManager().isIndexerIdle()) && (i < 100)));
    }
  }

  /**
   * Configure the include paths of a CDT project
   */
  public static boolean configureIncludes(final IProject project, final List<String> depPkgList) {
    boolean _xblockexpression = false;
    {
      final ICProjectDescriptionManager mngr = CoreModel.getDefault().getProjectDescriptionManager();
      ICProjectDescription cdesc = mngr.getProjectDescription(project, true);
      boolean _xifexpression = false;
      if ((cdesc != null)) {
        boolean _xtrycatchfinallyexpression = false;
        try {
          boolean _xblockexpression_1 = false;
          {
            ICConfigurationDescription[] _configurations = cdesc.getConfigurations();
            for (final ICConfigurationDescription configDescr : _configurations) {
              {
                final ICFolderDescription folderDescription = configDescr.getRootFolderDescription();
                final ICLanguageSetting[] languageSettings = folderDescription.getLanguageSettings();
                final ArrayList<ICLanguageSettingEntry> icIncludePaths = new ArrayList<ICLanguageSettingEntry>();
                CIncludePathEntry _cIncludePathEntry = new CIncludePathEntry("${workspace_loc:/${ProjName}/src-gen}", 0);
                icIncludePaths.add(_cIncludePathEntry);
                final String amentPrefixPath = EnvironmentUtils.get(Ros2Constants.AMENT_PREFIX_PATH);
                if ((amentPrefixPath != null)) {
                  String[] _split = amentPrefixPath.split(":");
                  for (final String amentEntry : _split) {
                    String _format = String.format("%s/include", amentEntry);
                    CIncludePathEntry _cIncludePathEntry_1 = new CIncludePathEntry(_format, 0);
                    icIncludePaths.add(_cIncludePathEntry_1);
                  }
                }
                for (final String depPkg : depPkgList) {
                  String _includePath = ProjectTools.getIncludePath(depPkg);
                  CIncludePathEntry _cIncludePathEntry_2 = new CIncludePathEntry(_includePath, 0);
                  icIncludePaths.add(_cIncludePathEntry_2);
                }
                for (final ICLanguageSetting lang : languageSettings) {
                  String[] _sourceExtensions = lang.getSourceExtensions();
                  for (final String ext : _sourceExtensions) {
                    boolean _equals = ext.equals("cpp");
                    if (_equals) {
                      lang.setSettingEntries(ICSettingEntry.INCLUDE_PATH, icIncludePaths);
                    }
                  }
                }
              }
            }
            mngr.setProjectDescription(project, cdesc, true, null);
            _xblockexpression_1 = ManagedBuildManager.saveBuildInfo(project, true);
          }
          _xtrycatchfinallyexpression = _xblockexpression_1;
        } catch (final Throwable _t) {
          if (_t instanceof CoreException) {
            final CoreException ce = (CoreException)_t;
            String _message = ce.getMessage();
            throw new RuntimeException(_message);
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
        _xifexpression = _xtrycatchfinallyexpression;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  public static void genCode(final RoboticsCppCreator codeGen, final org.eclipse.uml2.uml.Class component) {
    final UniqueEList<org.eclipse.uml2.uml.Package> packagesToGenerate = new UniqueEList<org.eclipse.uml2.uml.Package>();
    packagesToGenerate.add(component.getNearestPackage());
    EList<Classifier> _includedClassifiers = CppClassUtils.includedClassifiers(component);
    for (final Classifier incCl : _includedClassifiers) {
      boolean _isApplied = StereotypeUtil.isApplied(incCl, External.class);
      boolean _not = (!_isApplied);
      if (_not) {
        packagesToGenerate.add(incCl.getNearestPackage());
      }
    }
    for (final org.eclipse.uml2.uml.Package pkg : packagesToGenerate) {
      NullProgressMonitor _nullProgressMonitor = new NullProgressMonitor();
      codeGen.createPackageableElement(pkg, _nullProgressMonitor);
    }
  }

  /**
   * defaults to workspace path
   */
  public static String getIncludePath(final String pkgName) {
    final String prefixPath = EnvironmentUtils.get(Ros2Constants.AMENT_PREFIX_PATH);
    if ((prefixPath != null)) {
      String[] _split = prefixPath.split(":");
      for (final String pathEntry : _split) {
        {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append(pathEntry);
          _builder.append("/include/");
          _builder.append(pkgName);
          final String testPath = _builder.toString();
          boolean _exists = new File(testPath).exists();
          if (_exists) {
            return testPath;
          }
        }
      }
    }
    boolean _since = Ros2Distributions.since(Ros2Distributions.getRosDistro(), Ros2Distributions.RDLiteral.HUMBLE);
    if (_since) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("${WorkspaceDirPath}/install/");
      _builder.append(pkgName);
      _builder.append("/include/");
      _builder.append(pkgName);
      return _builder.toString();
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("${WorkspaceDirPath}/install/");
      _builder_1.append(pkgName);
      _builder_1.append("/include");
      return _builder_1.toString();
    }
  }
}
