/*****************************************************************************
 * Copyright (c) 2025 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr - Bug #22, dummy robot example is broken
 *          (better separate YAML file creation by moving into separate class)
 *
 *****************************************************************************/

package org.eclipse.papyrus.robotics.ros2.codegen.common.launch

import java.util.ArrayList
import java.util.List
import org.eclipse.papyrus.designer.deployment.tools.DepUtils
import org.eclipse.papyrus.robotics.bpc.profile.bpc.Entity
import org.eclipse.uml2.uml.Class
import org.eclipse.uml2.uml.InstanceValue
import org.eclipse.uml2.uml.NamedElement
import org.eclipse.uml2.uml.Property
import org.eclipse.uml2.uml.util.UMLUtil

import static extension org.eclipse.papyrus.robotics.core.utils.InstanceUtils.*
import static extension org.eclipse.papyrus.robotics.core.utils.ParameterUtils.getAllParameters
import static extension org.eclipse.papyrus.robotics.ros2.codegen.common.utils.SequencerUtils.*
import static extension org.eclipse.papyrus.robotics.ros2.codegen.common.utils.SkillUtils.*

/**
 * Create a set of launch scripts
 */
class ConfigYAML {

	/**
	 * initialize parameter values from instance specifications and the (implicit) system sequencer model
	 * (In the current implementation, the default skill semantics realization is always selected)
	 */
	static def createParameterFile(Class system) '''
		«FOR part : system.compInstanceList»
			«IF part.type instanceof Class && (part.type as Class).allParameters.size > 0»
				«part.name»:
				  ros__parameters:
				«val names = new ArrayList<String>()»
				«FOR param : (part.type as Class).allParameters»
					«param.hierarchicalName(names)» : «part.paramValue(param)»
				«ENDFOR»
			«ENDIF»
		«ENDFOR»
		«IF !system.uniqueSkills.nullOrEmpty»
			«system.sequencerName»:
			  ros__parameters:
			    default_bt_xml_filename: "$(find-pkg-share «system.getROS2PackageNameOfDefaultTask»)/«system.getPackageRelativeFolderNameOfDefaultTask»/«system.getBTFileNameOfDefaultTask»"
			    plugin_lib_names: [
			    «FOR skill : system.getUniqueSkills SEPARATOR ','»
			    	«"  \""+skill.realizationFileName»_bt_node"
			    «ENDFOR»
			    ]
		«ENDIF»
	'''

	/**
	 * Return either the specific value configured on the instance level or the default value
	 */
	static def paramValue(Property part, Property parameter) {
		var value = "";
		if (part.defaultValue instanceof InstanceValue) {
			val is = (part.defaultValue as InstanceValue).instance
			if (is !== null) {
				for (slot : is.slots) {
					if ( slot.definingFeature == parameter && DepUtils.firstValue(slot) !== null) {
						value = DepUtils.firstValue(slot).stringValue.quoteEmpty
					}
				}
			}
		}
		if (value.length == 0) {
			if (parameter.defaultValue !== null) {
				value = parameter.defaultValue.stringValue.quoteEmpty + " # (default)"
			}
			else {
				value = "'' # (empty)"			
			}
		}
		val entity = UMLUtil.getStereotypeApplication(parameter, Entity);
		if (entity !== null && entity.description !== null) {
			if (!value.contains("#")) {
				value += " #"
			}
			value += " " + entity.description
		}
		return value
	}
	
	/**
	 * #22: if a parameter or default parameter is empty, replace with 2 single quotes
	 */
	static def quoteEmpty(String value) {
		if (value.length == 0)
			return "''"
		else
			return value
	}
		
	/**
	 * Calculate the YAML representation of a hierarchical name with "." separator
	 * If the parent is used the first time, emit it
	 */
	static def hierarchicalName(NamedElement ne, List<String> names) {
		var result = ""
		// initial indentation is four spaces
		var indent = "    "
		var qName = ""
		for (String part : ne.name.split("\\.")) {
			qName += part + "."
			if (!names.contains(qName)) {
				names.add(qName);
				if (result.length > 4) {
					result += ":\n"
				}
				result += indent + part
			}
			indent += "  "
		}
		return result;
	}
}