/**
 * Copyright (c) 2018-2021 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.ros2.codegen.common.message;

import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.MARTE.MARTE_Annexes.VSL.DataTypes.ChoiceType;
import org.eclipse.papyrus.designer.infra.base.StringConstants;
import org.eclipse.papyrus.designer.languages.common.base.file.IPFileSystemAccess;
import org.eclipse.papyrus.robotics.bpc.profile.bpc.Entity;
import org.eclipse.papyrus.robotics.codegen.common.utils.PackageTools;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.MessageUtils;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.ParameterableElement;
import org.eclipse.uml2.uml.PrimitiveType;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.TemplateBinding;
import org.eclipse.uml2.uml.TemplateParameterSubstitution;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;

@SuppressWarnings("all")
public class CreateMessage {
  private CreateMsgPackage msgPkgCreator;

  /**
   * Construct
   */
  public CreateMessage(final CreateMsgPackage msgPkgCreator) {
    this.msgPkgCreator = msgPkgCreator;
  }

  /**
   * Create contents for a data type or an enumeration
   */
  public CharSequence createDtOrEnumMsg(final DataType dtOrEnum) {
    CharSequence _xifexpression = null;
    if ((dtOrEnum instanceof Enumeration)) {
      _xifexpression = this.createEnumMsgs(((Enumeration) dtOrEnum));
    } else {
      _xifexpression = this.createDataTypeMsg(dtOrEnum);
    }
    return _xifexpression;
  }

  /**
   * Create message contents for a dataType
   */
  public CharSequence createDataTypeMsg(final DataType dataType) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("# Publish-Subscribe (or Push, Send)");
    _builder.newLine();
    CharSequence _comment = this.comment(dataType);
    _builder.append(_comment);
    _builder.newLineIfNotEmpty();
    MessageUtils.makeDTExternal(dataType);
    _builder.newLineIfNotEmpty();
    {
      if ((StereotypeUtil.isApplied(dataType, ChoiceType.class) && false)) {
        _builder.append("# attributes of choice-type ");
        String _qualifiedName = dataType.getQualifiedName();
        _builder.append(_qualifiedName);
        _builder.append(" (only first attribute for the moment)");
        _builder.newLineIfNotEmpty();
        final Property attribute = dataType.getAllAttributes().get(0);
        _builder.newLineIfNotEmpty();
        _builder.append("\t\t\t");
        String _createAttributeMsg = this.createAttributeMsg(attribute);
        _builder.append(_createAttributeMsg, "\t\t\t");
        _builder.newLineIfNotEmpty();
      } else {
        _builder.append("# attributes of datatype ");
        String _qualifiedName_1 = dataType.getQualifiedName();
        _builder.append(_qualifiedName_1);
        _builder.newLineIfNotEmpty();
        {
          EList<Property> _allAttributes = dataType.getAllAttributes();
          for(final Property attribute_1 : _allAttributes) {
            String _createAttributeMsg_1 = this.createAttributeMsg(attribute_1);
            _builder.append(_createAttributeMsg_1);
            _builder.newLineIfNotEmpty();
          }
        }
      }
    }
    return _builder;
  }

  public CharSequence createServiceMsg(final DataType request, final DataType response) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("# query pattern - request");
    _builder.newLine();
    CharSequence _comment = this.comment(request);
    _builder.append(_comment);
    _builder.newLineIfNotEmpty();
    _builder.append("# attributes of request ");
    String _qualifiedName = request.getQualifiedName();
    _builder.append(_qualifiedName);
    _builder.newLineIfNotEmpty();
    {
      EList<Property> _allAttributes = request.getAllAttributes();
      for(final Property attribute : _allAttributes) {
        String _createAttributeMsg = this.createAttributeMsg(attribute);
        _builder.append(_createAttributeMsg);
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("---");
    _builder.newLine();
    CharSequence _comment_1 = this.comment(response);
    _builder.append(_comment_1);
    _builder.newLineIfNotEmpty();
    _builder.append("# attributes of response ");
    String _qualifiedName_1 = response.getQualifiedName();
    _builder.append(_qualifiedName_1);
    _builder.newLineIfNotEmpty();
    {
      EList<Property> _allAttributes_1 = response.getAllAttributes();
      for(final Property attribute_1 : _allAttributes_1) {
        String _createAttributeMsg_1 = this.createAttributeMsg(attribute_1);
        _builder.append(_createAttributeMsg_1);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  public CharSequence createActionMsg(final DataType goal, final DataType response, final DataType feedback) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("# action pattern");
    _builder.newLine();
    _builder.append("# attributes of goal ");
    String _qualifiedName = goal.getQualifiedName();
    _builder.append(_qualifiedName);
    _builder.newLineIfNotEmpty();
    {
      EList<Property> _allAttributes = goal.getAllAttributes();
      for(final Property attribute : _allAttributes) {
        String _createAttributeMsg = this.createAttributeMsg(attribute);
        _builder.append(_createAttributeMsg);
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("---");
    _builder.newLine();
    _builder.append("# attributes of result ");
    String _qualifiedName_1 = response.getQualifiedName();
    _builder.append(_qualifiedName_1);
    _builder.newLineIfNotEmpty();
    {
      EList<Property> _allAttributes_1 = response.getAllAttributes();
      for(final Property attribute_1 : _allAttributes_1) {
        String _createAttributeMsg_1 = this.createAttributeMsg(attribute_1);
        _builder.append(_createAttributeMsg_1);
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("---");
    _builder.newLine();
    _builder.append("# attributes of feedback ");
    String _qualifiedName_2 = feedback.getQualifiedName();
    _builder.append(_qualifiedName_2);
    _builder.newLineIfNotEmpty();
    {
      EList<Property> _allAttributes_2 = feedback.getAllAttributes();
      for(final Property attribute_2 : _allAttributes_2) {
        String _createAttributeMsg_2 = this.createAttributeMsg(attribute_2);
        _builder.append(_createAttributeMsg_2);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  /**
   * Create an attribute within a message, handling default values and comments
   */
  public String createAttributeMsg(final Property attribute) {
    String typeStr = null;
    Type _type = attribute.getType();
    if ((_type instanceof PrimitiveType)) {
      typeStr = MessageUtils.primitiveTypeMap(attribute.getType());
    } else {
      Type _type_1 = attribute.getType();
      if ((_type_1 instanceof Enumeration)) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("# enum ");
        String _name = attribute.getType().getName();
        _builder.append(_name);
        _builder.newLineIfNotEmpty();
        _builder.append("uint8");
        typeStr = _builder.toString();
        Type _type_2 = attribute.getType();
        final Enumeration type = ((Enumeration) _type_2);
        this.msgPkgCreator.createMsgPkg(type);
      } else {
        Type _type_3 = attribute.getType();
        if ((_type_3 instanceof DataType)) {
          Type _type_4 = attribute.getType();
          final DataType type_1 = ((DataType) _type_4);
          typeStr = String.format("%s/%s", PackageTools.pkgName(MessageUtils.getMessagePackage(attribute.getType())), attribute.getType().getName());
          this.msgPkgCreator.createMsgPkg(type_1);
        }
      }
    }
    StringConcatenation _builder_1 = new StringConcatenation();
    _builder_1.append(typeStr);
    String _multiplicityStr = this.getMultiplicityStr(attribute);
    _builder_1.append(_multiplicityStr);
    _builder_1.append(" ");
    String _name_1 = attribute.getName();
    _builder_1.append(_name_1);
    CharSequence _defaultValueStr = this.defaultValueStr(attribute);
    _builder_1.append(_defaultValueStr);
    CharSequence _comment = this.comment(attribute);
    _builder_1.append(_comment);
    _builder_1.newLineIfNotEmpty();
    return _builder_1.toString();
  }

  public CharSequence comment(final Element element) {
    final Entity entity = UMLUtil.<Entity>getStereotypeApplication(element, Entity.class);
    if ((((entity != null) && (entity.getDescription() != null)) && (entity.getDescription().length() > 0))) {
      String prefix = "";
      if ((element instanceof Property)) {
        prefix = "\t";
      }
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(prefix);
      _builder.append("# ");
      String _description = entity.getDescription();
      _builder.append(_description);
      return _builder;
    }
    return StringConstants.EMPTY;
  }

  public String getMultiplicityStr(final Property attribute) {
    int _upper = attribute.getUpper();
    boolean _equals = (_upper == (-1));
    if (_equals) {
      return "[]";
    } else {
      int _upper_1 = attribute.getUpper();
      boolean _notEquals = (_upper_1 != 1);
      if (_notEquals) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("[");
        int _upper_2 = attribute.getUpper();
        _builder.append(_upper_2);
        _builder.append("]");
        return _builder.toString();
      }
    }
    return null;
  }

  public CharSequence defaultValueStr(final Property attribute) {
    StringConcatenation _builder = new StringConcatenation();
    {
      ValueSpecification _defaultValue = attribute.getDefaultValue();
      boolean _tripleNotEquals = (_defaultValue != null);
      if (_tripleNotEquals) {
        _builder.append(" ");
        _builder.append("= ");
        String _stringValue = attribute.getDefaultValue().stringValue();
        _builder.append(_stringValue);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  /**
   * generate a message file, unless the message is an already existing system message
   * This is the case for messages/services that can be found with an empty AMENT_PREFIX_PATH.
   * The user can specify a configurable path that containing entries that should not be
   * re-written by code generation.
   */
  public void generateFile(final IPFileSystemAccess fileAccess, final String fileName, final String ext, final CharSequence content) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(ext);
    _builder.append("/");
    _builder.append(fileName);
    _builder.append(".");
    _builder.append(ext);
    final String fileNameWithExt = _builder.toString();
    fileAccess.generateFile(fileNameWithExt, content.toString());
  }

  public CharSequence createEnumMsgs(final Enumeration enumeration) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("# constants to mimic enumeration ");
    String _name = enumeration.getName();
    _builder.append(_name);
    _builder.newLineIfNotEmpty();
    int counter = 0;
    _builder.newLineIfNotEmpty();
    {
      EList<EnumerationLiteral> _ownedLiterals = enumeration.getOwnedLiterals();
      for(final EnumerationLiteral enumLiteral : _ownedLiterals) {
        _builder.append("uint8 ");
        String _upperCase = enumLiteral.getName().toUpperCase();
        _builder.append(_upperCase);
        _builder.append(" = ");
        int _plusPlus = counter++;
        _builder.append(_plusPlus);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  public CharSequence createMsgForEvent(final TemplateBinding tb) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("# Event");
    _builder.newLine();
    {
      EList<TemplateParameterSubstitution> _parameterSubstitutions = tb.getParameterSubstitutions();
      for(final TemplateParameterSubstitution tps : _parameterSubstitutions) {
        final ParameterableElement actual = tps.getActual();
        _builder.newLineIfNotEmpty();
        String _name = ((NamedElement) actual).getName();
        _builder.append(_name);
        _builder.append(" ");
        String _tPName = InteractionUtils.getTPName(tps.getFormal());
        _builder.append(_tPName);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }
}
