/*****************************************************************************
 * Copyright (c) 2019, 2023 CEA LIST and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Initial API and implementation
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Bug 568480
 *****************************************************************************/

package org.eclipse.papyrus.model2doc.emf.template2structure.generator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.Assert;
import org.eclipse.osgi.util.NLS;
import org.eclipse.papyrus.model2doc.core.generatorconfiguration.IDocumentStructureGeneratorConfiguration;
import org.eclipse.papyrus.model2doc.emf.documentstructuretemplate.DocumentTemplate;
import org.eclipse.papyrus.model2doc.emf.template2structure.Activator;
import org.eclipse.papyrus.model2doc.emf.template2structure.internal.generator.Template2StructureGeneratorDescription;
import org.eclipse.papyrus.model2doc.emf.template2structure.internal.generator.Template2StructureGeneratorExtensionReader;
import org.eclipse.papyrus.model2doc.emf.template2structure.mapping.AbstractTemplateToStructureMapper;

/**
 * Registry for StructureGenerator and mappers classes
 */
public final class Template2StructureRegistry {

	/**
	 * a map with the registered generator, by id
	 */
	private final Map<String, Template2StructureGeneratorDescription> generators;

	/**
	 * the singleton of this class
	 */
	public static final Template2StructureRegistry INSTANCE = new Template2StructureRegistry();

	/**
	 *
	 * Constructor.
	 *
	 */
	private Template2StructureRegistry() {
		this.generators = new Template2StructureGeneratorExtensionReader().readContributions();
	}


	/**
	 *
	 * @param generatorId
	 *            the id of a generator
	 * @return
	 *         the generator for this id or <code>null</code> if not found
	 */
	public final ITemplate2StructureGenerator getGenerator(final String generatorId) {
		Assert.isNotNull(generatorId);
		final ITemplate2StructureGenerator generator = this.generators.get(generatorId).getGenerator();
		if (null == generator) {
			Activator.log.warn(NLS.bind("The is no generator registered with the id {0}", generatorId)); //$NON-NLS-1$
		}
		return generator;
	}

	/**
	 *
	 * @param docTemplate
	 *            a document template
	 * @return
	 *         the generator if found, or <code>null</code>
	 */
	public ITemplate2StructureGenerator getGenerator(final DocumentTemplate docTemplate) {
		Assert.isNotNull(docTemplate);
		final IDocumentStructureGeneratorConfiguration conf = docTemplate.getDocumentStructureGeneratorConfiguration();
		if (null != conf) {
			return getGenerator(conf.getStructureGeneratorId());
		}
		return null;
	}

	/**
	 *
	 * @param docTemplate
	 *            a document template
	 * @return
	 *         the mappers registered for the generator declared in the document template
	 */
	public List<AbstractTemplateToStructureMapper<?>> getMappers(final DocumentTemplate docTemplate) {
		final String structureGeneratorId = docTemplate.getDocumentStructureGeneratorConfiguration().getStructureGeneratorId();
		if (null == structureGeneratorId || structureGeneratorId.isEmpty()) {
			Activator.log.warn(NLS.bind("The document structure generator id is not defined for {0}", docTemplate.toString())); //$NON-NLS-1$
			return Collections.emptyList();
		}

		final Template2StructureGeneratorDescription generatorDescription = this.generators.get(structureGeneratorId);
		if (generatorDescription == null) {
			Activator.log.warn(NLS.bind("There is no generator registered with the id {0}", structureGeneratorId)); //$NON-NLS-1$
			return Collections.emptyList();
		}
		return generatorDescription.getMappers();
	}

	/**
	 *
	 * @param docTemplate
	 *            a document template
	 * @return
	 *         the list of generators for this document template
	 */
	public List<ITemplate2StructureGenerator> getAvailableGenerators(final DocumentTemplate docTemplate) {
		final List<ITemplate2StructureGenerator> generators = new ArrayList<>();
		for (Template2StructureGeneratorDescription current : this.generators.values()) {
			if (current.getGenerator().handles(docTemplate)) {
				generators.add(current.getGenerator());
			}
		}
		return generators;
	}
}
