/**
 * Copyright (c) 2020 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.ros2.codegen.python.component;

import org.eclipse.papyrus.designer.languages.common.profile.Codegen.NoCodeGen;
import org.eclipse.papyrus.designer.languages.common.profile.Codegen.TemplateBinding;
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.papyrus.robotics.codegen.common.utils.ActivityUtils;
import org.eclipse.papyrus.robotics.codegen.common.utils.ApplyProfiles;
import org.eclipse.papyrus.robotics.core.utils.FunctionUtils;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.papyrus.robotics.profile.robotics.components.ActivityPort;
import org.eclipse.papyrus.robotics.profile.robotics.functions.FunctionKind;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.MessageUtils;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.RosHelpers;
import org.eclipse.papyrus.robotics.ros2.codegen.python.utils.RosPythonTypes;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.BehavioralFeature;
import org.eclipse.uml2.uml.PackageableElement;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.ParameterDirectionKind;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Type;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Exceptions;

/**
 * provide the callback functions (via bind) for the various kind of ports
 */
@SuppressWarnings("all")
public class Callbacks {
  private static final String PUSH = "PUSH or SEND";

  private static final String QUERY = "QUERY";

  private static final String ACTION = "ACTION";

  private static final String WHOLE_PKG = "Please note that code gets generated for the whole ROS 2 package, not only for the currently open component";

  /**
   * Return the of callback method (which corresponds to activity handling the port)
   * and provide additional parameter
   */
  public static String callBackMethodForPush(final org.eclipse.uml2.uml.Class component, final Port port) {
    final ActivityPort activity = ActivityUtils.getActivityForPort(component, port);
    Callbacks.checkActivity(activity, Callbacks.PUSH, port);
    final Behavior fct = FunctionUtils.getFunction(activity, FunctionKind.HANDLER);
    Callbacks.checkFunction(fct, Callbacks.PUSH, activity, port);
    final Behavior fctCopy = component.getOwnedBehavior(fct.getName());
    BehavioralFeature _specification = fctCopy.getSpecification();
    boolean _tripleNotEquals = (_specification != null);
    if (_tripleNotEquals) {
      fctCopy.getSpecification().createOwnedParameter("commobj", InteractionUtils.getCommObject(port));
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("self.");
      String _name = fct.getName();
      _builder.append(_name);
      return _builder.toString();
    }
    return null;
  }

  /**
   * Add parameters to return type
   */
  public static void clientCallBackMethodForService(final org.eclipse.uml2.uml.Class component, final Port port) {
    final ActivityPort activityPort = ActivityUtils.getActivityForPort(component, port);
    Callbacks.checkActivity(activityPort, Callbacks.QUERY, port);
    final Behavior resultFct = FunctionUtils.getFunction(activityPort, FunctionKind.HANDLER, "result");
    if ((resultFct != null)) {
      final Type clientType = RosPythonTypes.getType(port, "rclpy::client::Client");
      final Behavior lResultFct = component.getOwnedBehavior(resultFct.getName());
      final Parameter resultParam = lResultFct.getSpecification().createOwnedParameter("future", clientType);
      ApplyProfiles.applyCommonProfile(resultParam);
      final TemplateBinding resultTpl = StereotypeUtil.<TemplateBinding>applyApp(resultParam, TemplateBinding.class);
      resultTpl.getActuals().add(MessageUtils.getServiceType(port));
    }
  }

  /**
   * Return the callback method (which corresponds to activity handling of the port)
   * and provide additional parameter
   */
  public static String serverCallBackMethodForService(final org.eclipse.uml2.uml.Class component, final Port port) {
    final ActivityPort activityPort = ActivityUtils.getActivityForPort(component, port);
    Callbacks.checkActivity(activityPort, Callbacks.QUERY, port);
    final Behavior fct = FunctionUtils.getFunction(activityPort, FunctionKind.HANDLER);
    Callbacks.checkFunction(fct, Callbacks.QUERY, activityPort, port);
    final Behavior fctCopy = component.getOwnedBehavior(fct.getName());
    BehavioralFeature _specification = fctCopy.getSpecification();
    boolean _tripleNotEquals = (_specification != null);
    if (_tripleNotEquals) {
      fctCopy.getSpecification().createOwnedParameter("request", MessageUtils.getServiceType(port));
      fctCopy.getSpecification().createOwnedParameter("response", MessageUtils.getServiceType(port));
      final Parameter retParam = fctCopy.getSpecification().createOwnedParameter("ret", 
        RosHelpers.getRosPrimitiveType(port, "primitive::bool"));
      retParam.setDirection(ParameterDirectionKind.RETURN_LITERAL);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("self.");
      String _name = fct.getName();
      _builder.append(_name);
      return _builder.toString();
    }
    return null;
  }

  /**
   * Adapt client side callback methods, i.e. add necessary parameters to their signature.
   * The actual association with a specific call must be done by the user code during setup of
   * the call options.
   *  - The goal callback, that gets called if a client sets a new goal.
   *  - The cancel callback for canceling a previous goal
   *  - The accept callback to accept ... (What?)
   * 
   * @param component a component
   * @param port a port for which to provide callbacks
   */
  public static void clientCallsbacksForAction(final org.eclipse.uml2.uml.Class component, final Port port) {
    final ActivityPort activityPort = ActivityUtils.getActivityForPort(component, port);
    Callbacks.checkActivity(activityPort, Callbacks.ACTION, port);
    final Type handleType = RosPythonTypes.getType(port, "ros2Library::rclcpp_action::ClientGoalHandle");
    final Behavior feedbackFct = FunctionUtils.getFunction(activityPort, FunctionKind.HANDLER, FunctionUtils.R_FEEDBACK);
    if ((feedbackFct != null)) {
      final Behavior lFeedbackFct = component.getOwnedBehavior(feedbackFct.getName());
      final Parameter handleParam = lFeedbackFct.getSpecification().createOwnedParameter("handle", handleType);
      ApplyProfiles.applyCommonProfile(handleParam);
      final TemplateBinding template = StereotypeUtil.<TemplateBinding>applyApp(handleParam, TemplateBinding.class);
      template.getActuals().add(MessageUtils.getServiceType(port));
      lFeedbackFct.getSpecification().createOwnedParameter("feedback", MessageUtils.getServiceType(port));
    }
    final Behavior resultFct = FunctionUtils.getFunction(activityPort, FunctionKind.HANDLER, FunctionUtils.R_RESULT);
    if ((resultFct != null)) {
      final Behavior lResultFct = component.getOwnedBehavior(resultFct.getName());
      final Parameter resultParam = lResultFct.getSpecification().createOwnedParameter("result", handleType);
      final TemplateBinding resultTpl = StereotypeUtil.<TemplateBinding>applyApp(resultParam, TemplateBinding.class);
      resultTpl.getActuals().add(MessageUtils.getServiceType(port));
    }
    final Behavior goalFct = FunctionUtils.getFunction(activityPort, FunctionKind.HANDLER, FunctionUtils.GOAL);
    if ((goalFct != null)) {
      final Behavior lGoalFct = component.getOwnedBehavior(goalFct.getName());
      final String futureTypeName = String.format("std::shared_future<rclcpp_action::ClientGoalHandle<%s>::SharedPtr>", 
        RosHelpers.externalName(MessageUtils.getServiceType(port)));
      lGoalFct.getSpecification().createOwnedParameter("future", Callbacks.dummyType(component, futureTypeName));
    }
  }

  public static String serverCallsbacksForAction(final org.eclipse.uml2.uml.Class component, final Port port) {
    final ActivityPort activityPort = ActivityUtils.getActivityForPort(component, port);
    Callbacks.checkActivity(activityPort, Callbacks.ACTION, port);
    final Behavior goalFct = FunctionUtils.getFunction(activityPort, FunctionKind.HANDLER, FunctionUtils.GOAL);
    Callbacks.checkFunction(goalFct, (Callbacks.ACTION + "/feedback"), activityPort, port);
    final Behavior cancelFct = FunctionUtils.getFunction(activityPort, FunctionKind.HANDLER, FunctionUtils.P_CANCEL);
    Callbacks.checkFunction(cancelFct, (Callbacks.ACTION + "/result"), activityPort, port);
    final Behavior acceptedFct = FunctionUtils.getFunction(activityPort, FunctionKind.HANDLER, FunctionUtils.P_ACCEPTED);
    Callbacks.checkFunction(acceptedFct, (Callbacks.ACTION + "/goal"), activityPort, port);
    final Behavior lGoalFct = component.getOwnedBehavior(goalFct.getName());
    final Behavior lCancelFct = component.getOwnedBehavior(cancelFct.getName());
    final Behavior lAcceptedFct = component.getOwnedBehavior(acceptedFct.getName());
    final Type goalUUID = RosPythonTypes.getType(port, "ros2Library::rclcpp_action::GoalUUID");
    final Type goalResponse = RosPythonTypes.getType(port, "ros2Library::rclcpp_action::GoalResponse");
    final Type cancelResponse = RosPythonTypes.getType(port, "ros2Library::rclcpp_action::CancelResponse");
    lGoalFct.getSpecification().createOwnedParameter("uuid", goalUUID);
    final String goalTypeName = String.format("std::shared_ptr<const %s::Goal>", RosHelpers.externalName(MessageUtils.getServiceType(port)));
    lGoalFct.getSpecification().createOwnedParameter("goal", Callbacks.dummyType(component, goalTypeName));
    final Parameter goalRetParam = lGoalFct.getSpecification().createOwnedParameter("return", goalResponse);
    goalRetParam.setDirection(ParameterDirectionKind.RETURN_LITERAL);
    final String goalHandleTypeName = String.format("const std::shared_ptr<rclcpp_action::ServerGoalHandle<%s>>", 
      RosHelpers.externalName(MessageUtils.getServiceType(port)));
    lCancelFct.getSpecification().createOwnedParameter("goal_handle", Callbacks.dummyType(component, goalHandleTypeName));
    final Parameter cancelRetParam = lCancelFct.getSpecification().createOwnedParameter("return", cancelResponse);
    cancelRetParam.setDirection(ParameterDirectionKind.RETURN_LITERAL);
    lAcceptedFct.getSpecification().createOwnedParameter("goal_handle", Callbacks.dummyType(component, goalHandleTypeName));
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("std::bind(&");
    String _qualifiedName = component.getQualifiedName();
    _builder.append(_qualifiedName);
    String _postfix = ActivityUtils.getPostfix(component);
    _builder.append(_postfix);
    _builder.append("::");
    String _name = goalFct.getName();
    _builder.append(_name);
    _builder.append(", (");
    String _name_1 = component.getName();
    _builder.append(_name_1);
    String _postfix_1 = ActivityUtils.getPostfix(component);
    _builder.append(_postfix_1);
    _builder.append("*) this, std::placeholders::_1, std::placeholders::_2),");
    _builder.newLineIfNotEmpty();
    _builder.append("std::bind(&");
    String _qualifiedName_1 = component.getQualifiedName();
    _builder.append(_qualifiedName_1);
    String _postfix_2 = ActivityUtils.getPostfix(component);
    _builder.append(_postfix_2);
    _builder.append("::");
    String _name_2 = cancelFct.getName();
    _builder.append(_name_2);
    _builder.append(", (");
    String _name_3 = component.getName();
    _builder.append(_name_3);
    String _postfix_3 = ActivityUtils.getPostfix(component);
    _builder.append(_postfix_3);
    _builder.append("*) this, std::placeholders::_1),");
    _builder.newLineIfNotEmpty();
    _builder.append("std::bind(&");
    String _qualifiedName_2 = component.getQualifiedName();
    _builder.append(_qualifiedName_2);
    String _postfix_4 = ActivityUtils.getPostfix(component);
    _builder.append(_postfix_4);
    _builder.append("::");
    String _name_4 = acceptedFct.getName();
    _builder.append(_name_4);
    _builder.append(", (");
    String _name_5 = component.getName();
    _builder.append(_name_5);
    String _postfix_5 = ActivityUtils.getPostfix(component);
    _builder.append(_postfix_5);
    _builder.append("*) this, std::placeholders::_1)");
    _builder.newLineIfNotEmpty();
    return _builder.toString();
  }

  /**
   * Utility function for error handling
   */
  public static void checkActivity(final ActivityPort activityPort, final String portKind, final Port port) {
    try {
      if ((activityPort == null)) {
        String _format = String.format(("The %s port \"%s\" of component \"%s\" is not connected with any activity port. " + Callbacks.WHOLE_PKG), portKind, port.getName(), port.getClass_().getName());
        throw new TransformationException(_format);
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Utility function for error handling
   */
  public static void checkFunction(final Behavior function, final String portKind, final ActivityPort activity, final Port port) {
    try {
      if ((function == null)) {
        String _format = String.format(("No handler function (for %s) is found for activity port \"%s\" associated with port \"%s\" of component \"%s\". " + Callbacks.WHOLE_PKG), portKind, activity.getBase_Port().getName(), port.getName(), port.getClass_().getName());
        throw new TransformationException(_format);
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Create a dummy type with a certain name (since some complex parameter signatures
   * can not be expressed using standard mechanisms
   */
  public static org.eclipse.uml2.uml.Class dummyType(final org.eclipse.uml2.uml.Class component, final String name) {
    PackageableElement _packagedElement = component.getNearestPackage().getPackagedElement("dummytypes");
    org.eclipse.uml2.uml.Package pkg = ((org.eclipse.uml2.uml.Package) _packagedElement);
    if ((pkg == null)) {
      pkg = component.getNearestPackage().createNestedPackage("dummytypes");
      ApplyProfiles.applyCommonProfile(pkg);
      StereotypeUtil.apply(pkg, NoCodeGen.class);
    }
    PackageableElement _packagedElement_1 = pkg.getPackagedElement(name);
    org.eclipse.uml2.uml.Class dummyType = ((org.eclipse.uml2.uml.Class) _packagedElement_1);
    if ((dummyType == null)) {
      dummyType = pkg.createOwnedClass(name, false);
    }
    return dummyType;
  }
}
