/*****************************************************************************
 * Copyright (c) 2020 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 *
 *****************************************************************************/

package org.eclipse.papyrus.robotics.ros2.codegen.component

import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.ConstInit
import org.eclipse.papyrus.robotics.core.utils.PortUtils
import org.eclipse.papyrus.robotics.ros2.codegen.utils.ApplyProfiles
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil
import org.eclipse.uml2.uml.Class
import org.eclipse.uml2.uml.Classifier
import org.eclipse.uml2.uml.OpaqueBehavior
import org.eclipse.uml2.uml.Operation
import org.eclipse.uml2.uml.Port
import org.eclipse.uml2.uml.UMLPackage
import org.eclipse.uml2.uml.profile.standard.Create

import static extension org.eclipse.papyrus.robotics.core.utils.InteractionUtils.*
import static extension org.eclipse.papyrus.robotics.ros2.codegen.component.Callbacks.*
import static extension org.eclipse.papyrus.robotics.ros2.codegen.utils.Helpers.*
import static extension org.eclipse.papyrus.robotics.ros2.codegen.launch.LaunchScript.*
import static extension org.eclipse.papyrus.robotics.ros2.codegen.utils.MessageUtils.*
import org.eclipse.papyrus.robotics.profile.robotics.components.ComponentPort
import org.eclipse.uml2.uml.util.UMLUtil

/**
 * Manage constructor creation, including port
 */
class Constructor {

	def static createConstructor(Class component) {
		val lcNodeSC = getRosType(component, "ros2Library::rclcpp_lifecycle::LifecycleNode")
		if (lcNodeSC instanceof Classifier) {
			component.createGeneralization(lcNodeSC)
		}
		val op = addConstrOp(component);
		addConstrMethod(component, op);
	}

	/**
	 * Add an init operation (constructor) that creates the ROS2 publishers/subscribers/clients/... for ports 
	 */
	def static addConstrOp(Class component) {
		// val nodeSC = getRosType(component, "ros2Library::rclcpp::Node");
		val nodeOptions = getRosType(component, "ros2Library::rclcpp::NodeOptions")
		// val string = getPrimitiveType(component, "PrimitiveTypes::String")
		// component.createOwnedAttribute("instName", string)
		val init = component.createOwnedOperation(component.name, null, null)
		init.createOwnedParameter("options", nodeOptions)
		var create = StereotypeUtil.applyApp(init, Create)
		if (create === null) {
			ApplyProfiles.applyStdProfile(init)
			create = StereotypeUtil.applyApp(init, Create)
		}
		var constInit = StereotypeUtil.applyApp(init, ConstInit)
		if (constInit === null) {
			ApplyProfiles.applyCppProfile(init)
			constInit = StereotypeUtil.applyApp(init, ConstInit)
		}
		constInit.initialisation = '''rclcpp_lifecycle::LifecycleNode("«component.name»", options)'''
		return init
	}

	/**
	 * Add a method body to the constructor operation
	 */
	def static addConstrMethod(Class component, Operation constructorOp) {
		val ob = component.createOwnedBehavior(component.name, UMLPackage.eINSTANCE.getOpaqueBehavior) as OpaqueBehavior
		constructorOp.methods.add(ob)
		ob.languages.add("C++")
		ob.bodies.add('''
			«FOR port : PortUtils.getAllPorts(component)»
				«IF port.communicationPattern !== null»
					«val pattern = port.communicationPattern»
					«IF pattern.isPush»
						«port.createPush»

					«ELSEIF pattern.isSend»
						«port.createSend»

					«ELSEIF pattern.isQuery»
						«port.createQuery»

					«ELSEIF pattern.isAction»
						«port.createAction»

					«ELSEIF pattern.isEvent»
						««« TODO: unsupported
					«ENDIF»
				«ENDIF»
			«ENDFOR»
		''')
	}

	/**
	 * Create a publisher or subscriber
	 */
	def static createPush(Port port) '''
		«IF port.provideds.size() > 0»
««« Publisher
			«port.name»_pub_ = create_publisher<«port.commObject.externalName»>("«port.topic»",
					«port.qos("1")»);
			// directly activate a publisher
			«port.name»_pub_->on_activate();
		«ELSEIF port.requireds.size() > 0»
««« Subscriber
			«val defaultQoS = "rclcpp::QoS(rclcpp::KeepLast(100)).best_effort()"»
				«port.name»_sub_ = create_subscription<«port.commObject.externalName»>("«port.topic»", «port.qos(defaultQoS)», «port.class_.callBackMethodForPush(port)»);
		«ENDIF»
	'''

	/**
	 * Create send (publisher and subscriber as well)
	 */
	def static createSend(Port port) '''
		«IF port.provideds.size() > 0»
««« Subscriber
			«val defaultQoS = "rclcpp::QoS(rclcpp::KeepLast(100)).best_effort()"»
				«port.name»_recv_ = create_subscription<«port.commObject.externalName»>("«port.topic»", «port.qos(defaultQoS)», «port.class_.callBackMethodForPush(port)»);
		«ELSEIF port.requireds.size() > 0»
««« Publisher
			«port.name»_send_ = create_publisher<«port.commObject.externalName»>("«port.topic»",
					«port.qos("1")»);
			// directly activate a publisher
			«port.name»_send_->on_activate();
		«ENDIF»
	'''

	/**
	 * Create a service client or server
	 */
	def static createQuery(Port port) '''
		«IF port.provideds.size() > 0»
			««« Server
			«port.name»_srv_ = create_service<«port.serviceType.externalName»>("«port.serviceType.nameWoPrefix»", «port.class_.serverCallBackMethodForService(port)»);
		«ELSEIF port.requireds.size() > 0»
			«port.name»_client_ = create_client<«port.serviceType.externalName»>("«port.serviceType.nameWoPrefix»");
			«port.class_.clientCallBackMethodForService(port)»
		«ENDIF»
	'''

	/**
	 * Create an action client or server
	 * TODO: code currently only copied from Fibonacci example.
	 */
	def static createAction(Port port) '''
		«IF port.provideds.size() > 0»
			«port.name»_actsrv_ = rclcpp_action::create_server<«port.serviceType.externalName»>(
				this->get_node_base_interface(),
				this->get_node_clock_interface(),
				this->get_node_logging_interface(),
				this->get_node_waitables_interface(),
				"«port.topic»",
				«port.class_.serverCallsbacksForAction(port)»);
		«ELSE»
			«port.name»_actcli_ = rclcpp_action::create_client<«port.serviceType.externalName»>(
				this->get_node_base_interface(),
				this->get_node_graph_interface(),
				this->get_node_logging_interface(),
				this->get_node_waitables_interface(),
				"«port.topic»");
			«port.class_.clientCallsbacksForAction(port)»
		«ENDIF»
	'''

	/**
	 * Create an event client and server
	 * TODO - not supported
	 */
	def static createEvent(Port port) '''
«««		This pattern is aimed to create run-time configurable notification mechanism.
«««		The client can register to be notified when a particular event happens on the server side.
«««		The server catches the events and sends a message only to clients interested to that
«««		particular event.
«««		The event condition check on the server side must be written by user, the pattern specifies
«««		the semantic of the <RequestedEvent, ReplyMessage, DataToCheck>
«««		There is no equivalent pattern in YARP, but it can be implemented via specialized
«««		client/server devices.
	'''

	/**
	 * Return the QoS associated with a port or defaultQoS, if none is specified
	 * 
	 * The "KEEP_LAST" history setting tells DDS to store a fixed-size buffer of values before they
	 * are sent, to aid with recovery in the event of dropped messages.
	 * use best effort to avoid blocking during execution.
	 */	
	def static qos(Port port, String defaultQoS) {
		val compPort = UMLUtil.getStereotypeApplication(port, ComponentPort)
		if (compPort !== null && compPort.qos !== null && compPort.qos.length > 0) {
			return compPort.qos
		}
		else {
			return defaultQoS
		}
 	}
}
