/*****************************************************************************
 * Copyright (c) 2017 CEA LIST and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Nicolas FAUVERGUE (CEA LIST) nicolas.fauvergue@cea.fr - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrus.interoperability.sysml14.sysml.xmi.helper;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.papyrus.sysml14.modelelements.Conform;
import org.eclipse.papyrus.sysml14.modelelements.Stakeholder;
import org.eclipse.papyrus.sysml14.modelelements.View;
import org.eclipse.papyrus.sysml14.modelelements.Viewpoint;
import org.eclipse.papyrus.uml.tools.utils.UMLUtil;
import org.eclipse.uml2.uml.Comment;
import org.eclipse.uml2.uml.Dependency;
import org.eclipse.uml2.uml.Operation;

/**
 * This class defines the needed methods to preserve or create the SysML XMI identifiers.
 */
public class SysMLXMIIDHelper {

	/**
	 * The base_Class identifier.
	 */
	private static final String BASE_CLASS = "baseClass"; //$NON-NLS-1$

	/**
	 * The base_Classifier identifier.
	 */
	private static final String BASE_CLASSIFIER = "baseClassifier"; //$NON-NLS-1$

	/**
	 * The base_Generalization identifier.
	 */
	private static final String BASE_GENERALIZATION = "baseGeneralization"; //$NON-NLS-1$

	/**
	 * The comment identifier.
	 */
	private static final String COMMENT = "comment"; //$NON-NLS-1$

	/**
	 * The concernList identifier.
	 */
	private static final String CONCERNLIST = "concernList_"; //$NON-NLS-1$

	/**
	 * The createOperation identifier.
	 */
	private static final String CREATEOPERATION = "createOperation"; //$NON-NLS-1$

	/**
	 * The stakeholder identifier.
	 */
	private static final String STAKEHOLDER = "stakeholder"; //$NON-NLS-1$

	/**
	 * The 'Of' separator in identifier.
	 */
	private static final String OF = "_Of_"; //$NON-NLS-1$

	/**
	 * This allows to calculate the identifiers of converted or created SysML elements.
	 * 
	 * @param res
	 *            The UML Resource.
	 * @param current
	 *            The object to manage its identifier.
	 * @param currentId
	 *            The identifier of existing element.
	 * @param oldIds
	 *            The existing objects and their identifiers.
	 */
	public static void calculateMoreNeededId(final XMIResource res, final Object current, final String currentId, final Map<EObject, String> oldIds) {
		if (current instanceof View) {
			calculateBaseClassViewId(res, (View) current, currentId, oldIds);
		} else if (current instanceof Conform) {
			calculateBaseGeneralizationConformId(res, (Conform) current, currentId, oldIds);
		} else if (current instanceof Viewpoint) {
			calculateOthersViewPointId(res, (Viewpoint) current, currentId, oldIds);
		}
	}

	/**
	 * This allows to calculate the identifiers of converted or created 'View'.
	 * 
	 * @param res
	 *            The UML Resource.
	 * @param view
	 *            The View to manage its identifier.
	 * @param currentId
	 *            The identifier of existing element.
	 * @param oldIds
	 *            The existing objects and their identifiers.
	 */
	public static void calculateBaseClassViewId(final XMIResource res, final View view, final String currentId, final Map<EObject, String> oldIds) {

		String basePackageId = null;

		final EObject basedClass = view.getBase_Class();
		EStructuralFeature basePackageFeature = null;
		Iterator<EStructuralFeature> features = view.eClass().getEAllStructuralFeatures().iterator();
		while (features.hasNext() && null == basePackageFeature) {
			EStructuralFeature feature = features.next();
			if (feature.getName().equals("base_Package")) { //$NON-NLS-1$
				basePackageFeature = feature;
			}
		}

		if (null != basePackageFeature) {
			final Object basePackage = view.eGet(basePackageFeature);
			if (basePackage instanceof Package && oldIds.containsKey(basePackage)) {
				basePackageId = oldIds.get(basePackage);
			}
		}

		// Manage the base class created
		if (null != basePackageId) {
			res.setID(basedClass, basePackageId);
		} else {
			final StringBuilder stringBuilder = new StringBuilder();
			stringBuilder.append(BASE_CLASS);
			stringBuilder.append(OF);
			stringBuilder.append(currentId);
			res.setID(basedClass, stringBuilder.toString());
			oldIds.put(basedClass, stringBuilder.toString());
		}
	}

	/**
	 * This allows to calculate the identifiers of converted or created 'Conform'.
	 * 
	 * @param res
	 *            The UML Resource.
	 * @param conform
	 *            The Conform to manage its identifier.
	 * @param currentId
	 *            The identifier of existing element.
	 * @param oldIds
	 *            The existing objects and their identifiers.
	 */
	public static void calculateBaseGeneralizationConformId(final XMIResource res, final Conform conform, final String currentId, final Map<EObject, String> oldIds) {

		String basePackageId = null;

		final EObject basedGeneralization = conform.getBase_Generalization();
		EStructuralFeature baseDependencyFeature = null;
		Iterator<EStructuralFeature> features = conform.eClass().getEAllStructuralFeatures().iterator();
		while (features.hasNext() && null == baseDependencyFeature) {
			EStructuralFeature feature = features.next();
			if (feature.getName().equals("base_Dependency")) { //$NON-NLS-1$
				baseDependencyFeature = feature;
			}
		}

		if (null != baseDependencyFeature) {
			final Object baseDependency = conform.eGet(baseDependencyFeature);
			if (baseDependency instanceof Dependency && oldIds.containsKey(baseDependency)) {
				basePackageId = oldIds.get(baseDependency);
			}
		}

		// Manage the base generalization created
		if (null != basePackageId) {
			res.setID(basedGeneralization, basePackageId);
		} else {
			final StringBuilder stringBuilder = new StringBuilder();
			stringBuilder.append(BASE_GENERALIZATION);
			stringBuilder.append(OF);
			stringBuilder.append(currentId);
			res.setID(basedGeneralization, stringBuilder.toString());
			oldIds.put(basedGeneralization, stringBuilder.toString());
		}
	}

	/**
	 * This allows to calculate the identifiers of converted or created 'Viewpoint'.
	 * 
	 * @param res
	 *            The UML Resource.
	 * @param viewpoint
	 *            The Viewpoint to manage its identifier.
	 * @param currentId
	 *            The identifier of existing element.
	 * @param oldIds
	 *            The existing objects and their identifiers.
	 */
	public static void calculateOthersViewPointId(final XMIResource res, final Viewpoint viewpoint, final String currentId, final Map<EObject, String> oldIds) {
		final org.eclipse.uml2.uml.Class basedClass = viewpoint.getBase_Class();

		// Manage the new concern list comments
		int concernListIndex = 1;
		for (final Comment comment : viewpoint.getConcernList()) {
			if (!oldIds.containsKey(comment)) {
				final StringBuilder stringBuilder = new StringBuilder();
				stringBuilder.append(CONCERNLIST);
				stringBuilder.append(concernListIndex);
				stringBuilder.append(OF);
				stringBuilder.append(currentId);
				res.setID(comment, stringBuilder.toString());
				oldIds.put(comment, stringBuilder.toString());
				concernListIndex++;
			}
		}

		// Manage the created operation 'create'
		Iterator<Operation> operations = basedClass.getOperations().iterator();
		boolean foundCreate = false;
		while (operations.hasNext() && !foundCreate) {
			final Operation operation = operations.next();
			if (null != UMLUtil.getAppliedStereotype(operation, "StandardProfile::Create", false) && !oldIds.containsKey(operation)) { //$NON-NLS-1$
				foundCreate = true;
				final StringBuilder stringBuilder = new StringBuilder();
				stringBuilder.append(CREATEOPERATION);
				stringBuilder.append(OF);
				stringBuilder.append(currentId);
				res.setID(operation, stringBuilder.toString());
				oldIds.put(operation, stringBuilder.toString());

				if (operation.getOwnedComments().size() == 1) {
					// Manage comment in 'create' operation
					Comment comment = operation.getOwnedComments().get(0);
					final StringBuilder commentStringBuilder = new StringBuilder();
					commentStringBuilder.append(COMMENT);
					commentStringBuilder.append(OF);
					commentStringBuilder.append(stringBuilder);
					res.setID(comment, commentStringBuilder.toString());
					oldIds.put(comment, commentStringBuilder.toString());
				}
			}
		}

		// Manage the stakeholders
		int stakeholderIndex = 1;
		final List<Stakeholder> stakeholders = new ArrayList<Stakeholder>(viewpoint.getStakeholder());
		// Sort the list of stakeholders to be sure about the order of named elements
		Collections.sort(stakeholders, new Comparator<Stakeholder>() {
			@Override
			public int compare(Stakeholder stakeholder1, Stakeholder stakeholder2) {
				return stakeholder1.getBase_Classifier().getName().compareTo(stakeholder2.getBase_Classifier().getName());
			}
		});
		for (final Stakeholder stakeholder : stakeholders) {
			final StringBuilder stringBuilder = new StringBuilder();
			stringBuilder.append(STAKEHOLDER);
			stringBuilder.append(stakeholderIndex);
			stringBuilder.append(OF);
			stringBuilder.append(currentId);
			res.setID(stakeholder, stringBuilder.toString());
			oldIds.put(stakeholder, stringBuilder.toString());

			// Manage the reference class
			final EObject nestedClassifier = stakeholder.getBase_Classifier();
			final StringBuilder otherStringBuilder = new StringBuilder();
			otherStringBuilder.append(BASE_CLASSIFIER);
			otherStringBuilder.append(OF);
			otherStringBuilder.append(stringBuilder);
			res.setID(nestedClassifier, otherStringBuilder.toString());
			oldIds.put(nestedClassifier, otherStringBuilder.toString());

			stakeholderIndex++;
		}
	}

	/**
	 * This allows to determinate if the created stereotype application need to be managed.
	 * 
	 * @param current
	 *            The stereotype application.
	 * @return <code>true</code> if the stereotype application identifier need to be managed, <code>false</code> otherwise.
	 */
	public static boolean isStereotypedElementCreated(final EObject current) {
		// Need to be implemented for the SysML stereotype created element
		return false;
	}

	/**
	 * This allows to calculate the identifiers of the stereotype applications.
	 * 
	 * @param res
	 *            The UML Resource.
	 * @param current
	 *            The stereotype application.
	 */
	public static void calculateCreateStereotypedElement(final XMIResource res, final EObject current) {
		// Need to be implemented for the SysML stereotype created element
	}

}
