/*****************************************************************************
 * Copyright (c) 2017 CEA LIST and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * 	 Maged Elaasar - Initial API and Implementation
 *
 *****************************************************************************/
package org.eclipse.papyrus.sysml.architecture.migration;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gmf.runtime.common.core.command.AbstractCommand;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.papyrus.infra.architecture.ArchitectureDomainManager;
import org.eclipse.papyrus.infra.core.architecture.RepresentationKind;
import org.eclipse.papyrus.infra.core.architecture.merged.MergedArchitectureDescriptionLanguage;
import org.eclipse.papyrus.infra.nattable.common.reconciler.TableReconciler;
import org.eclipse.papyrus.infra.nattable.model.nattable.Table;
import org.eclipse.papyrus.infra.nattable.representation.PapyrusTable;
import org.eclipse.papyrus.sysml.architecture.SysMLArchitectureContextIds;

/**
 * UML Table Reconciler from 1.0.0 to 1.1.0 that replaces the old prototype value for tables
 * with ones based on table kinds
 */
public class SysMLTableReconciler_1_3_0 extends TableReconciler {

	private static final String ALLOCATION_TABLE = "Allocation Table";
	private static final String ALLOCATION_TABLE_URI = "org.eclipse.papyrus.sysml.nattable.allocation.config/resources/allocationTable.configuration";
	private static final String REQUIREMENT_TABLE = "Requirement Table";
	private static final String REQUIREMENT_TABLE_URI = "org.eclipse.papyrus.sysml.nattable.requirement.config/configs/requirementTable.configuration";

	@Override
	public ICommand getReconcileCommand(Table table) {
		RepresentationKind newTableKind = null;
		if (table.getPrototype() instanceof org.eclipse.papyrus.infra.viewpoints.configuration.PapyrusView) {
			org.eclipse.papyrus.infra.viewpoints.configuration.PapyrusView oldTableKind = (org.eclipse.papyrus.infra.viewpoints.configuration.PapyrusView) table.getPrototype();
			if (oldTableKind.eIsProxy()) {
				if (EcoreUtil.getURI(oldTableKind).toString().contains(ALLOCATION_TABLE_URI)) {
					newTableKind = getTableKind(ALLOCATION_TABLE, table);
				} else if (EcoreUtil.getURI(oldTableKind).toString().contains(REQUIREMENT_TABLE_URI)) {
					newTableKind = getTableKind(REQUIREMENT_TABLE, table);
				}
			}
		}
		if (newTableKind != null)
			return new ReplaceTablePrototypeCommand(table, newTableKind);
		return null;
	}

	protected PapyrusTable getTableKind(String name, Table table) {
		ArchitectureDomainManager manager = ArchitectureDomainManager.getInstance();
		MergedArchitectureDescriptionLanguage context = (MergedArchitectureDescriptionLanguage) manager.getArchitectureContextById(SysMLArchitectureContextIds.SysML);
		for (RepresentationKind pKind : context.getRepresentationKinds()) {
			if (pKind.getName().equals(name)) {
				PapyrusTable tKind = (PapyrusTable) pKind;
				if (tKind.getModelRules().get(0).getElement().isInstance(table.getContext())) {
					return tKind;
				}
			}
		}
		return null;
	}

	/**
	 * A command to replace the old table prototype with the new representation kinds
	 */
	protected class ReplaceTablePrototypeCommand extends AbstractCommand {

		private Table table;
		private RepresentationKind newKind;

		public ReplaceTablePrototypeCommand(Table table, RepresentationKind newKind) {
			super("Replace the papyrus view style from 1.0.0 to 1.1.0");
			this.table = table;
			this.newKind = newKind;
		}

		@Override
		protected CommandResult doExecuteWithResult(IProgressMonitor progressMonitor, IAdaptable info) throws ExecutionException {
			table.setPrototype(null);
			table.setTableKindId(newKind.getId());
			return CommandResult.newOKCommandResult();
		}

		@Override
		public boolean canUndo() {
			return false;
		}

		@Override
		public boolean canRedo() {
			return false;
		}

		@Override
		protected CommandResult doRedoWithResult(IProgressMonitor progressMonitor, IAdaptable info) throws ExecutionException {
			throw new ExecutionException("Should not be called, canRedo false"); //$NON-NLS-1$
		}

		@Override
		protected CommandResult doUndoWithResult(IProgressMonitor progressMonitor, IAdaptable info) throws ExecutionException {
			throw new ExecutionException("Should not be called, canUndo false"); //$NON-NLS-1$
		}
	}
}
