/**
 * Copyright (c) 2015 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.designer.components.transformation.cpp.xtend;

import com.google.common.base.Objects;
import java.util.HashMap;
import java.util.Map;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.components.modellibs.core.transformations.Constants;
import org.eclipse.papyrus.designer.components.transformation.PortInfo;
import org.eclipse.papyrus.designer.components.transformation.PortUtils;
import org.eclipse.papyrus.designer.components.transformation.component.PrefixConstants;
import org.eclipse.papyrus.designer.components.transformation.cpp.Messages;
import org.eclipse.papyrus.designer.components.transformation.extensions.IOOTrafo;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Ptr;
import org.eclipse.papyrus.designer.transformation.base.utils.CopyUtils;
import org.eclipse.papyrus.designer.transformation.base.utils.PartsUtil;
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException;
import org.eclipse.papyrus.designer.transformation.core.transformations.LazyCopier;
import org.eclipse.papyrus.designer.uml.tools.utils.ConnectorUtil;
import org.eclipse.papyrus.designer.uml.tools.utils.ElementUtils;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.AggregationKind;
import org.eclipse.uml2.uml.Association;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.ConnectableElement;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.InterfaceRealization;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.xtend2.lib.StringConcatenation;

/**
 * This class realizes the dynamic variant of the OO-transformation
 * 
 * TODO: currently not tested/used, needs to be aligned with abstract transformation
 *       In does currently not support multiple references, as the static variant
 */
@SuppressWarnings("all")
public class DynamicCppToOO implements IOOTrafo {
  protected LazyCopier copier;

  private static final String PART_MANAGER = "services::PartManager";

  private static final String INIT_PARTS = "initParts";

  private static final String PARTS = "parts";

  private static final String progLang = "C/C++";

  protected org.eclipse.uml2.uml.Class bootloader;

  @Override
  public void init(final LazyCopier copier, final org.eclipse.uml2.uml.Class bootloader) {
    PrefixConstants.init(PrefixConstants.CIFvariant.UML);
    this.copier = copier;
    this.bootloader = bootloader;
  }

  @Override
  public void addPortOperations(final org.eclipse.uml2.uml.Class implementation) {
    this.addGetPortOperation(implementation);
    DynamicCppToOO.addConnectPortOperation(implementation);
  }

  /**
   * Add the get_p operation for each port with a provided interface. It also
   * adds a suitable implementation that evaluates delegation connectors from
   * the port to a property within the composite. The delegation target could
   * either be a normal class (no port) or an inner component.
   * 
   * @param implementation
   */
  public void addGetPortOperation(final org.eclipse.uml2.uml.Class implementation) {
    EList<PortInfo> _flattenExtendedPorts = PortUtils.flattenExtendedPorts(PortUtils.getAllPorts2(implementation));
    for (final PortInfo portInfo : _flattenExtendedPorts) {
      {
        final Interface providedIntf = portInfo.getProvided();
        if ((providedIntf != null)) {
          String _name = portInfo.getName();
          final String opName = (PrefixConstants.getP_Prefix + _name);
          Operation op = implementation.getOwnedOperation(opName, null, null);
          if ((op != null)) {
            Type _type = op.getType();
            boolean _notEquals = (!Objects.equal(_type, providedIntf));
            if (_notEquals) {
              op.createOwnedParameter(Constants.retParamName, providedIntf);
            }
          } else {
            op = implementation.createOwnedOperation(opName, null, null, providedIntf);
            final Parameter retParam = op.getOwnedParameters().get(0);
            retParam.setName(Constants.retParamName);
            StereotypeUtil.apply(retParam, Ptr.class);
            Behavior _createOwnedBehavior = implementation.createOwnedBehavior(opName, UMLPackage.eINSTANCE.getOpaqueBehavior());
            final OpaqueBehavior behavior = ((OpaqueBehavior) _createOwnedBehavior);
            op.getMethods().add(behavior);
            final ConnectorEnd ce = ConnectorUtil.getDelegation(implementation, portInfo.getModelPort());
            String body = null;
            if ((ce != null)) {
              final Property part = ce.getPartWithPort();
              final ConnectableElement role = ce.getRole();
              body = "return ";
              if ((role instanceof Port)) {
                String _body = body;
                StringConcatenation _builder = new StringConcatenation();
                String _nameRef = CppUtils.nameRef(part);
                _builder.append(_nameRef);
                _builder.append(PrefixConstants.getP_Prefix);
                String _name_1 = ((Port)role).getName();
                _builder.append(_name_1);
                _builder.append("();");
                body = (_body + _builder);
              } else {
                String _body_1 = body;
                String _name_2 = role.getName();
                body = (_body_1 + _name_2);
              }
            } else {
              InterfaceRealization _interfaceRealization = implementation.getInterfaceRealization(null, providedIntf);
              boolean implementsIntf = (_interfaceRealization != null);
              if ((!implementsIntf)) {
                final Interface providedIntfInCopy = this.copier.<Interface>getCopy(providedIntf);
                InterfaceRealization _interfaceRealization_1 = implementation.getInterfaceRealization(null, providedIntfInCopy);
                boolean _tripleNotEquals = (_interfaceRealization_1 != null);
                implementsIntf = _tripleNotEquals;
              }
              if (implementsIntf) {
                body = "return this;";
              } else {
                String _format = String.format(Messages.CompImplTrafos_IntfNotImplemented, providedIntf.getName(), 
                  portInfo.getPort().getName(), implementation.getName());
                throw new RuntimeException(_format);
              }
            }
            behavior.getLanguages().add(DynamicCppToOO.progLang);
            behavior.getBodies().add(body);
          }
        }
      }
    }
  }

  /**
   * Add a connect_<portName> operation for ports with a required interface.
   * Whereas operation and a behavior is added for each owned port, a behavior
   * (method) is needed for ports inherited from a component type (the
   * behavior is implementation specific, as it needs to take delegation to
   * parts into account)
   * 
   * @param implementation
   */
  public static void addConnectPortOperation(final org.eclipse.uml2.uml.Class implementation) {
    EList<PortInfo> _flattenExtendedPorts = PortUtils.flattenExtendedPorts(PortUtils.getAllPorts2(implementation));
    for (final PortInfo portInfo : _flattenExtendedPorts) {
      {
        final Interface requiredIntf = portInfo.getRequired();
        if ((requiredIntf != null)) {
          String _name = portInfo.getName();
          final String opName = (PrefixConstants.connectQ_Prefix + _name);
          Operation _ownedOperation = implementation.getOwnedOperation(opName, null, null);
          boolean _tripleNotEquals = (_ownedOperation != null);
          if (_tripleNotEquals) {
          } else {
            Operation op = implementation.createOwnedOperation(opName, null, null);
            final Parameter refParam = op.createOwnedParameter("ref", requiredIntf);
            StereotypeUtil.apply(refParam, Ptr.class);
            Behavior _createOwnedBehavior = implementation.createOwnedBehavior(opName, UMLPackage.eINSTANCE.getOpaqueBehavior());
            final OpaqueBehavior behavior = ((OpaqueBehavior) _createOwnedBehavior);
            op.getMethods().add(behavior);
            final ConnectorEnd ce = ConnectorUtil.getDelegation(implementation, portInfo.getModelPort());
            String body = null;
            if ((ce != null)) {
              final Property part = ce.getPartWithPort();
              body = part.getName();
              final ConnectableElement role = ce.getRole();
              if ((role instanceof Port)) {
                String _name_1 = ((Port)role).getName();
                final String targetOpName = (PrefixConstants.connectQ_Prefix + _name_1);
                StringConcatenation _builder = new StringConcatenation();
                String _nameRef = CppUtils.nameRef(part);
                _builder.append(_nameRef);
                _builder.append(targetOpName);
                _builder.append("(ref)");
                body = _builder.toString();
              } else {
                String _body = body;
                StringConcatenation _builder_1 = new StringConcatenation();
                String _name_2 = part.getName();
                _builder_1.append(_name_2);
                _builder_1.append(";");
                body = (_body + _builder_1);
              }
            } else {
              String _name_3 = portInfo.getName();
              final String attributeName = (PrefixConstants.attributePrefix + _name_3);
              Property attr = implementation.getOwnedAttribute(attributeName, null);
              if (((attr == null) || (attr instanceof Port))) {
                attr = implementation.createOwnedAttribute(attributeName, requiredIntf);
                CopyUtils.copyMultElemModifiers(portInfo.getPort(), attr);
                attr.setAggregation(AggregationKind.SHARED_LITERAL);
              }
              body = attributeName;
              String _body_1 = body;
              body = (_body_1 + " = ref;");
            }
            behavior.getLanguages().add(DynamicCppToOO.progLang);
            behavior.getBodies().add(body);
            if (((PrefixConstants.getConnQ_Prefix.length() > 0) && (ce != null))) {
              String _name_4 = portInfo.getName();
              final String getConnOpName = (PrefixConstants.getConnQ_Prefix + _name_4);
              Operation getConnOp = implementation.getOwnedOperation(getConnOpName, null, null);
              if ((getConnOp == null)) {
                getConnOp = implementation.createOwnedOperation(getConnOpName, null, null, requiredIntf);
                final Parameter retParam = op.getOwnedParameters().get(0);
                retParam.setName(Constants.retParamName);
                StereotypeUtil.apply(retParam, Ptr.class);
              }
              Behavior _createOwnedBehavior_1 = implementation.createOwnedBehavior(getConnOpName, 
                UMLPackage.eINSTANCE.getOpaqueBehavior());
              final OpaqueBehavior getConnBehavior = ((OpaqueBehavior) _createOwnedBehavior_1);
              getConnOp.getMethods().add(getConnBehavior);
              String _name_5 = portInfo.getName();
              final String name = (PrefixConstants.attributePrefix + _name_5);
              StringConcatenation _builder_2 = new StringConcatenation();
              _builder_2.append("return ");
              _builder_2.append(name);
              _builder_2.append(";");
              body = _builder_2.toString();
              behavior.getLanguages().add(DynamicCppToOO.progLang);
              behavior.getBodies().add(body);
            }
          }
        }
      }
    }
  }

  /**
   * Add an operation "createConnections" that implements the connections
   * between composite parts. It only takes the assembly connections into
   * account, since delegation connectors are handled by the get_ and connect_
   * port operations above.
   * 
   * @param implementation
   */
  @Override
  public void addConnectionOperation(final org.eclipse.uml2.uml.Class compositeImplementation) throws TransformationException {
    String createConnBody = "";
    final Map<ConnectorEnd, Integer> indexMap = new HashMap<ConnectorEnd, Integer>();
    EList<Connector> _ownedConnectors = compositeImplementation.getOwnedConnectors();
    for (final Connector connector : _ownedConnectors) {
      boolean _isAssembly = ConnectorUtil.isAssembly(connector);
      if (_isAssembly) {
        int _size = connector.getEnds().size();
        boolean _notEquals = (_size != 2);
        if (_notEquals) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Connector <");
          String _name = connector.getName();
          _builder.append(_name);
          _builder.append("> does not have two ends. This is currently not supported");
          throw new TransformationException(_builder.toString());
        }
        final ConnectorEnd end1 = connector.getEnds().get(0);
        final ConnectorEnd end2 = connector.getEnds().get(1);
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("// realization of connector <");
        String _name_1 = connector.getName();
        _builder_1.append(_name_1);
        _builder_1.append(">\\n");
        String cmd = _builder_1.toString();
        if (((end1.getRole() instanceof Port) && PortUtils.isExtendedPort(((Port) end1.getRole())))) {
          ConnectableElement _role = end1.getRole();
          final Port port = ((Port) _role);
          final EList<PortInfo> subPorts = PortUtils.flattenExtendedPort(port);
          for (final PortInfo subPort : subPorts) {
            {
              String _cmd = cmd;
              StringConcatenation _builder_2 = new StringConcatenation();
              _builder_2.append("  ");
              _builder_2.append("// realization of connection for sub-port ");
              String _name_2 = subPort.getPort().getName();
              _builder_2.append(_name_2, "  ");
              _builder_2.append("\\n");
              cmd = (_cmd + _builder_2);
              String _cmd_1 = cmd;
              String _connectPorts = DynamicCppToOO.connectPorts(indexMap, connector, end1, end2, subPort.getPort());
              cmd = (_cmd_1 + _connectPorts);
              String _cmd_2 = cmd;
              String _connectPorts_1 = DynamicCppToOO.connectPorts(indexMap, connector, end2, end1, subPort.getPort());
              cmd = (_cmd_2 + _connectPorts_1);
            }
          }
        } else {
          String _cmd = cmd;
          String _connectPorts = DynamicCppToOO.connectPorts(indexMap, connector, end1, end2, null);
          cmd = (_cmd + _connectPorts);
          String _cmd_1 = cmd;
          String _connectPorts_1 = DynamicCppToOO.connectPorts(indexMap, connector, end2, end1, null);
          cmd = (_cmd_1 + _connectPorts_1);
        }
        String _createConnBody = createConnBody;
        createConnBody = (_createConnBody + (cmd + "\n"));
      }
    }
    int _length = createConnBody.length();
    boolean _greaterThan = (_length > 0);
    if (_greaterThan) {
      final Operation operation = compositeImplementation.createOwnedOperation(Constants.CREATE_CONNECTIONS, null, null);
      Behavior _createOwnedBehavior = compositeImplementation.createOwnedBehavior(operation.getName(), 
        UMLPackage.eINSTANCE.getOpaqueBehavior());
      final OpaqueBehavior behavior = ((OpaqueBehavior) _createOwnedBehavior);
      behavior.getLanguages().add(DynamicCppToOO.progLang);
      behavior.getBodies().add(createConnBody);
      behavior.setSpecification(operation);
    }
  }

  /**
   * Create the body C++ code code that creates a connection between the two ends
   * of a connector. This function checks whether the first end really is a receptacle
   * and the second really is a facet.
   * TODO: cleaner rewrite in xtend
   * 
   * @param indexMap
   *            a map of indices that are used in case of multiplex
   *            receptacles
   * @param connector
   *            a connector
   * @param receptacleEnd
   *            an end of the connector that may point to a receptacle port
   * @param facetEnd
   *            an end of the connector that may point to a facet port
   * @param subPort
   *            a sub-port in case of extended ports
   * @return
   * @throws TransformationException
   */
  public static String connectPorts(final Map<ConnectorEnd, Integer> indexMap, final Connector connector, final ConnectorEnd receptacleEnd, final ConnectorEnd facetEnd, final Port subPort) throws TransformationException {
    final Association association = connector.getType();
    if (((receptacleEnd.getRole() instanceof Port) && (facetEnd.getRole() instanceof Port))) {
      ConnectableElement _role = facetEnd.getRole();
      final Port facetPort = ((Port) _role);
      ConnectableElement _role_1 = receptacleEnd.getRole();
      final Port receptaclePort = ((Port) _role_1);
      final PortInfo facetPI = PortInfo.fromSubPort(facetPort, subPort);
      final PortInfo receptaclePI = PortInfo.fromSubPort(receptaclePort, subPort);
      if (((facetPI.getProvided() != null) && (receptaclePI.getRequired() != null))) {
        final Property facetPart = facetEnd.getPartWithPort();
        final Property receptaclePart = receptacleEnd.getPartWithPort();
        String subPortName = "";
        if ((subPort != null)) {
          String _subPortName = subPortName;
          String _name = subPort.getName();
          String _plus = ("_" + _name);
          subPortName = (_subPortName + _plus);
        }
        final String indexName = DynamicCppToOO.getIndexName(indexMap, receptaclePort, receptacleEnd);
        StringConcatenation _builder = new StringConcatenation();
        String _nameRef = CppUtils.nameRef(receptaclePart);
        _builder.append(_nameRef);
        _builder.append("connect_");
        String _name_1 = receptaclePort.getName();
        _builder.append(_name_1);
        _builder.append(" ");
        _builder.append(subPortName);
        _builder.append(";");
        final String setter = _builder.toString();
        StringConcatenation _builder_1 = new StringConcatenation();
        String _nameRef_1 = CppUtils.nameRef(facetPart);
        _builder_1.append(_nameRef_1);
        _builder_1.append("get_");
        String _name_2 = facetPort.getName();
        _builder_1.append(_name_2);
        _builder_1.append(" ");
        _builder_1.append(subPortName);
        _builder_1.append("()");
        final String getter = _builder_1.toString();
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append(setter);
        _builder_2.append("(");
        _builder_2.append(indexName);
        _builder_2.append(getter);
        _builder_2.append(");\\n");
        return _builder_2.toString();
      }
    } else {
      ConnectableElement _role_2 = receptacleEnd.getRole();
      if ((_role_2 instanceof Port)) {
        ConnectableElement _role_3 = receptacleEnd.getRole();
        final Port receptaclePort_1 = ((Port) _role_3);
        Interface _required = PortUtils.getRequired(receptaclePort_1);
        boolean _tripleNotEquals = (_required != null);
        if (_tripleNotEquals) {
          ConnectableElement _role_4 = facetEnd.getRole();
          final Property facetPart_1 = ((Property) _role_4);
          final Property receptaclePart_1 = facetEnd.getPartWithPort();
          final String indexName_1 = DynamicCppToOO.getIndexName(indexMap, receptaclePort_1, receptacleEnd);
          StringConcatenation _builder_3 = new StringConcatenation();
          String _nameRef_2 = CppUtils.nameRef(receptaclePart_1);
          _builder_3.append(_nameRef_2);
          _builder_3.append("connect_");
          String _name_3 = receptaclePort_1.getName();
          _builder_3.append(_name_3);
          final String setter_1 = _builder_3.toString();
          StringConcatenation _builder_4 = new StringConcatenation();
          _builder_4.append("&");
          String _name_4 = facetPart_1.getName();
          _builder_4.append(_name_4);
          final String getter_1 = _builder_4.toString();
          StringConcatenation _builder_5 = new StringConcatenation();
          _builder_5.append(setter_1);
          _builder_5.append("(");
          _builder_5.append(indexName_1);
          _builder_5.append(getter_1);
          _builder_5.append(");\\n");
          return _builder_5.toString();
        }
      } else {
        ConnectableElement _role_5 = facetEnd.getRole();
        if ((_role_5 instanceof Port)) {
          ConnectableElement _role_6 = facetEnd.getRole();
          final Port facetPort_1 = ((Port) _role_6);
          Interface _provided = PortUtils.getProvided(facetPort_1);
          boolean _tripleNotEquals_1 = (_provided != null);
          if (_tripleNotEquals_1) {
            final Property facetPart_2 = facetEnd.getPartWithPort();
            ConnectableElement _role_7 = facetEnd.getRole();
            final Property receptaclePart_2 = ((Property) _role_7);
            final String setter_2 = receptaclePart_2.getName();
            StringConcatenation _builder_6 = new StringConcatenation();
            String _nameRef_3 = CppUtils.nameRef(facetPart_2);
            _builder_6.append(_nameRef_3);
            _builder_6.append("get_");
            String _name_5 = facetPort_1.getName();
            _builder_6.append(_name_5);
            _builder_6.append("();");
            final String getter_2 = _builder_6.toString();
            StringConcatenation _builder_7 = new StringConcatenation();
            _builder_7.append(setter_2);
            _builder_7.append(" = ");
            _builder_7.append(getter_2);
            _builder_7.append(";\\n");
            return _builder_7.toString();
          }
        } else {
          if ((association != null)) {
            ConnectableElement _role_8 = facetEnd.getRole();
            final Property facetPart_3 = ((Property) _role_8);
            ConnectableElement _role_9 = receptacleEnd.getRole();
            final Property receptaclePart_3 = ((Property) _role_9);
            final Property assocProp1 = association.getMemberEnd(null, facetPart_3.getType());
            if (((assocProp1 != null) && assocProp1.isNavigable())) {
              StringConcatenation _builder_8 = new StringConcatenation();
              String _nameRef_4 = CppUtils.nameRef(receptaclePart_3);
              _builder_8.append(_nameRef_4);
              String _name_6 = assocProp1.getName();
              _builder_8.append(_name_6);
              final String setter_3 = _builder_8.toString();
              StringConcatenation _builder_9 = new StringConcatenation();
              _builder_9.append("&");
              String _name_7 = facetPart_3.getName();
              _builder_9.append(_name_7);
              final String getter_3 = _builder_9.toString();
              StringConcatenation _builder_10 = new StringConcatenation();
              _builder_10.append(setter_3);
              _builder_10.append(" = ");
              _builder_10.append(getter_3);
              _builder_10.append(";\\n");
              return _builder_10.toString();
            }
          } else {
            String _name_8 = connector.getName();
            String _plus_1 = ("Connector <" + _name_8);
            String _plus_2 = (_plus_1 + 
              "> does not use ports, but it is not typed (only connectors between ports should not be typed)");
            throw new TransformationException(_plus_2);
          }
        }
      }
    }
    return "";
  }

  /**
   * Handle ports with multiplicity > 1. The idea is that we could have
   * multiple connections targeting a receptacle. The first connection would
   * start with index 0. Implementations can make no assumption which
   * connection is associated with a certain index. [want to avoid associative
   * array in runtime].
   * 
   * @param port
   * @param end
   * @return
   */
  public static String getIndexName(final Map<ConnectorEnd, Integer> indexMap, final Port port, final ConnectorEnd end) {
    if (((port.getUpper() > 1) || (port.getUpper() == (-1)))) {
      Integer indexValue = indexMap.get(end);
      if ((indexValue == null)) {
        indexValue = Integer.valueOf(0);
        indexMap.put(end, indexValue);
      }
      String index = (indexValue + ", ");
      indexValue++;
      indexMap.put(end, indexValue);
      return index;
    }
    return "";
  }

  /**
   * Transform parts if necessary.
   * 
   * @param compositeImplementation
   *            a (composite) component
   */
  @Override
  public void transformParts(final org.eclipse.uml2.uml.Class compositeImplementation) {
    String initPartsBody = "";
    EList<Property> _parts = PartsUtil.getParts(compositeImplementation);
    for (final Property attribute : _parts) {
      {
        final Type type = attribute.getType();
        if ((type instanceof org.eclipse.uml2.uml.Class)) {
          String _initPartsBody = initPartsBody;
          String _initPartBody = this.initPartBody(attribute);
          initPartsBody = (_initPartsBody + _initPartBody);
          attribute.destroy();
        }
      }
    }
    final NamedElement partManager = ElementUtils.getQualifiedElementFromRS(compositeImplementation, DynamicCppToOO.PART_MANAGER);
    if ((partManager instanceof Type)) {
      compositeImplementation.createOwnedAttribute(DynamicCppToOO.PARTS, ((Type)partManager));
    }
    final Operation operation = compositeImplementation.createOwnedOperation(DynamicCppToOO.INIT_PARTS, null, null);
    String _name = operation.getName();
    String _plus = ("b:" + _name);
    Behavior _createOwnedBehavior = compositeImplementation.createOwnedBehavior(_plus, UMLPackage.eINSTANCE.getOpaqueBehavior());
    final OpaqueBehavior behavior = ((OpaqueBehavior) _createOwnedBehavior);
    behavior.getLanguages().add(DynamicCppToOO.progLang);
    behavior.getBodies().add(initPartsBody);
  }

  public String initPartBody(final Property part) {
    String _name = part.getName();
    String _plus = ("parts.add(" + _name);
    String _plus_1 = (_plus + ", ");
    Type _type = part.getType();
    String _plus_2 = (_plus_1 + _type);
    return (_plus_2 + ")");
  }
}
