/*****************************************************************************
 * Copyright (c) 2017 CEA LIST and Thales
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr (based on similar file
 *  							from Remi Schnekenburger)
 *
 *****************************************************************************/

package org.eclipse.papyrus.designer.ucm.core.commands;

import java.util.Collections;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand;
import org.eclipse.papyrus.designer.ucm.core.Activator;
import org.eclipse.papyrus.designer.ucm.core.Messages;
import org.eclipse.papyrus.designer.ucm.profile.UCMProfile.ucm_commons.ApplicationModule;
import org.eclipse.papyrus.designer.ucm.profile.UCMProfile.util.UCMResource;
import org.eclipse.papyrus.designer.ucm.supplement.profile.UCM_Supplement.util.UCMSuppResource;
import org.eclipse.papyrus.designer.ucm.types.profile.UCMTypes.util.UCMTypesResource;
import org.eclipse.papyrus.infra.architecture.commands.IModelCreationCommand;
import org.eclipse.papyrus.infra.core.resource.ModelSet;
import org.eclipse.papyrus.infra.emf.gmf.command.GMFtoEMFCommandWrapper;
import org.eclipse.papyrus.uml.tools.model.UmlUtils;
import org.eclipse.papyrus.uml.tools.utils.PackageUtil;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.UMLFactory;


/**
 * The Class CreateSysMLModelCommand.
 */
public class CreateUCMModelCommand implements IModelCreationCommand {

	@Override
	public void createModel(final ModelSet modelSet) {
		runAsTransaction(modelSet);
	}

	protected void runAsTransaction(final ModelSet modelSet) {
		// Get the UML element to which the newly created diagram will be
		// attached.
		// Create the diagram
		final Resource modelResource = UmlUtils.getUmlResource(modelSet);
		TransactionalEditingDomain editingDomain = modelSet.getTransactionalEditingDomain();

		AbstractTransactionalCommand command = new AbstractTransactionalCommand(editingDomain, Messages.CreateUCMModelCommand_InitModel, Collections.EMPTY_LIST) {

			@Override
			protected CommandResult doExecuteWithResult(IProgressMonitor monitor, IAdaptable info) throws ExecutionException {
				EObject model = getRootElement(modelResource);
				attachModelToResource(model, modelResource);

				initializeModel(model);
				return CommandResult.newOKCommandResult();

			}
		};
		editingDomain.getCommandStack().execute(new GMFtoEMFCommandWrapper(command));
	}

	protected EObject getRootElement(Resource modelResource) {
		EObject rootElement = null;
		if (modelResource != null && modelResource.getContents() != null && modelResource.getContents().size() > 0) {
			Object root = modelResource.getContents().get(0);
			if (root instanceof EObject) {
				rootElement = (EObject) root;
			}
		} else {
			rootElement = createRootElement();
		}
		return rootElement;
	}

	protected EObject createRootElement() {
		return UMLFactory.eINSTANCE.createModel();
	}

	protected void attachModelToResource(EObject root, Resource resource) {
		resource.getContents().add(root);
	}

	protected void initializeModel(EObject owner) {
		((org.eclipse.uml2.uml.Package) owner).setName(getModelName());

		// Retrieve UCM profile and apply with Sub-profiles
		Profile profile = (Profile) PackageUtil.loadPackage(UCMResource.PROFILE_PATH_URI, owner.eResource().getResourceSet());
		if (profile != null) {
			PackageUtil.applyProfile(((org.eclipse.uml2.uml.Package) owner), profile, true);
		} else {
			Activator.log.error(Messages.CreateUCMModelCommand_ProfileNotFound, null);
		}
		// Retrieve UCM profile and apply with Sub-profiles
		profile = (Profile) PackageUtil.loadPackage(UCMSuppResource.PROFILE_PATH_URI, owner.eResource().getResourceSet());
		if (profile != null) {
			PackageUtil.applyProfile(((org.eclipse.uml2.uml.Package) owner), profile, true);
		} else {
			Activator.log.error(Messages.CreateUCMModelCommand_ProfileSuppNotFound, null);
		}
		// Retrieve UCM profile and apply with Sub-profiles
		profile = (Profile) PackageUtil.loadPackage(UCMTypesResource.PROFILE_PATH_URI, owner.eResource().getResourceSet());
		if (profile != null) {
			PackageUtil.applyProfile(((org.eclipse.uml2.uml.Package) owner), profile, true);
		} else {
			Activator.log.error(Messages.CreateUCMModelCommand_ProfileTypesNotFound, null);
		}
		
		// apply ApplicationModule stereotype on root
		StereotypeUtil.apply((org.eclipse.uml2.uml.Element) owner, ApplicationModule.class);
	}

	/**
	 * Gets the model name.
	 *
	 * @return the model name
	 */
	protected String getModelName() {
		return "UCMModel"; //$NON-NLS-1$
	}
}
