/**
 * Copyright (c) 2016 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Van Cam Pham (CEA LIST) <vancam.pham@cea.fr> - Initial API and implementation
 */
package org.eclipse.papyrus.designer.languages.cpp.reverse.utils;

import com.google.common.collect.Iterables;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.eclipse.cdt.core.model.ICProject;
import org.eclipse.cdt.core.model.IParent;
import org.eclipse.cdt.core.model.ITranslationUnit;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Listens to changes in files
 */
@SuppressWarnings("all")
public class FileWatcher {
  private ICProject m_cproject;
  
  private long timestamp;
  
  public FileWatcher(final ICProject project, final long timestamp) {
    this.m_cproject = project;
    this.timestamp = timestamp;
  }
  
  public boolean isModified(final ITranslationUnit itu) {
    File file = itu.getFile().getRawLocation().makeAbsolute().toFile();
    Path filePath = file.toPath();
    BasicFileAttributes attributes = null;
    try {
      attributes = Files.<BasicFileAttributes>readAttributes(filePath, BasicFileAttributes.class);
    } catch (final Throwable _t) {
      if (_t instanceof IOException) {
        final IOException exception = (IOException)_t;
        String _message = exception.getMessage();
        String _plus = (("Exception handled when trying to get file " + 
          "attributes: ") + _message);
        System.out.println(_plus);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    long seconds = attributes.lastModifiedTime().to(TimeUnit.SECONDS);
    if ((((seconds > Long.MIN_VALUE) && (seconds < Long.MAX_VALUE)) && (seconds > this.timestamp))) {
      return true;
    }
    return false;
  }
  
  public List<ITranslationUnit> getModifiledTranslationUnits(final IParent parent) {
    try {
      final List<ITranslationUnit> ret = new UniqueEList<ITranslationUnit>();
      if ((!(parent instanceof ITranslationUnit))) {
        final Function1<ITranslationUnit, Boolean> _function = new Function1<ITranslationUnit, Boolean>() {
          @Override
          public Boolean apply(final ITranslationUnit it) {
            return Boolean.valueOf(FileWatcher.this.isModified(it));
          }
        };
        Iterables.<ITranslationUnit>addAll(ret, IterableExtensions.<ITranslationUnit>filter(Iterables.<ITranslationUnit>filter(((Iterable<?>)Conversions.doWrapArray(parent.getChildren())), ITranslationUnit.class), _function));
        final Consumer<IParent> _function_1 = new Consumer<IParent>() {
          @Override
          public void accept(final IParent it) {
            ret.addAll(FileWatcher.this.getModifiledTranslationUnits(it));
          }
        };
        Iterables.<IParent>filter(((Iterable<?>)Conversions.doWrapArray(parent.getChildren())), IParent.class).forEach(_function_1);
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
