/*******************************************************************************
 * Copyright (c) 2008 - 2010 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.util;

import org.eclipse.emf.mint.IItemJavaElementDescriptor;
import org.eclipse.emf.mint.IItemJavaElementDescriptor2;
import org.eclipse.emf.mint.IJavaTypeReference;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;

/**
 * Default implementation of {@link IItemJavaElementDescriptor}.
 * 
 * <p>
 * Clients may extend or instantiate this class.
 * </p>
 * 
 * @since 0.9
 */
public class ItemJavaElementDescriptor implements IItemJavaElementDescriptor2 {

	private final Kind kind;

	private final IJavaElement javaElement;

	private final IJavaTypeReference javaTypeReference;

	private final Object nonJavaElement;

	private final String category;

	private final String description;

	private final String displayName;

	private final Object feature;

	private final boolean elementOwner;

	/**
	 * Creates an instance with a specific Java element. All other arguments are
	 * optional.
	 * 
	 * @param javaElement
	 *            the described Java element
	 * @param category
	 *            category used to group descriptors
	 * @param description
	 *            description
	 * @param displayName
	 *            display name
	 * @param feature
	 *            related feature, if any
	 */
	public ItemJavaElementDescriptor(IJavaElement javaElement, String category,
			String description, String displayName, Object feature) {
		this(javaElement, category, description, displayName, feature, false);
	}

	/**
	 * Creates an instance with a specific Java element. All other arguments are
	 * optional.
	 * 
	 * @param javaElement
	 *            the described Java element
	 * @param category
	 *            category used to group descriptors
	 * @param description
	 *            description
	 * @param displayName
	 *            display name
	 * @param feature
	 *            related feature, if any
	 * @param elementOwner
	 *            <tt>true</tt> if the object is the owner of the referenced
	 *            element
	 * @since 0.9
	 */
	public ItemJavaElementDescriptor(IJavaElement javaElement, String category,
			String description, String displayName, Object feature,
			boolean elementOwner) {
		this(Kind.JAVA_ELEMENT, javaElement, null, null, category, description,
				displayName, feature, elementOwner);
	}

	/**
	 * Creates an instance with an unresolved reference to a Java type in the
	 * context of a project. All other arguments are optional.
	 * 
	 * @param context
	 *            Java project in whose context the type should be resolvable
	 * @param typeName
	 *            name of the referenced Java type
	 * @param category
	 *            category used to group descriptors
	 * @param description
	 *            description
	 * @param displayName
	 *            display name
	 * @param feature
	 *            related feature, if any
	 */
	public ItemJavaElementDescriptor(IJavaProject context, String typeName,
			String category, String description, String displayName,
			Object feature) {
		this(context, typeName, category, description, displayName, feature,
				false);
	}

	/**
	 * Creates an instance with an unresolved reference to a Java type in the
	 * context of a project. All other arguments are optional.
	 * 
	 * @param context
	 *            Java project in whose context the type should be resolvable
	 * @param typeName
	 *            name of the referenced Java type
	 * @param category
	 *            category used to group descriptors
	 * @param description
	 *            description
	 * @param displayName
	 *            display name
	 * @param feature
	 *            related feature, if any
	 * @param elementOwner
	 *            <tt>true</tt> if the object is the owner of the referenced
	 *            element
	 * @since 0.9
	 */
	public ItemJavaElementDescriptor(IJavaProject context, String typeName,
			String category, String description, String displayName,
			Object feature, boolean elementOwner) {
		this(Kind.JAVA_TYPE_REFERENCE, null, new JavaTypeReference(context,
				typeName), null, category, description, displayName, feature,
				elementOwner);
	}

	/**
	 * Creates an instance with a non-Java element. All other arguments are
	 * optional.
	 * 
	 * @param nonJavaElement
	 *            non-Java element
	 * @param category
	 *            category used to group descriptors
	 * @param description
	 *            description
	 * @param displayName
	 *            display name
	 * @param feature
	 *            related feature, if any
	 */
	public ItemJavaElementDescriptor(Object nonJavaElement, String category,
			String description, String displayName, Object feature) {
		this(nonJavaElement, category, description, displayName, feature, false);
	}

	/**
	 * Creates an instance with a non-Java element. All other arguments are
	 * optional.
	 * 
	 * @param nonJavaElement
	 *            non-Java element
	 * @param category
	 *            category used to group descriptors
	 * @param description
	 *            description
	 * @param displayName
	 *            display name
	 * @param feature
	 *            related feature, if any
	 * @param elementOwner
	 *            <tt>true</tt> if the object is the owner of the referenced
	 *            element
	 * @since 0.9
	 */
	public ItemJavaElementDescriptor(Object nonJavaElement, String category,
			String description, String displayName, Object feature,
			boolean elementOwner) {
		this(Kind.NON_JAVA_RESOURCE, null, null, nonJavaElement, category,
				description, displayName, feature, elementOwner);
	}

	/**
	 * Convenience constructor for setting all encapsulated fields in one place.
	 * 
	 * @param kind
	 *            element kind
	 * @param javaElement
	 *            java element (either the element or the reference are
	 *            required)
	 * @param javaTypeReference
	 *            java type reference (either the reference or the element are
	 *            required)
	 * @param nonJavaElement
	 *            non-Java element
	 * @param category
	 *            category
	 * @param description
	 *            description
	 * @param displayName
	 *            display name
	 * @param feature
	 *            related feature, if any
	 */
	protected ItemJavaElementDescriptor(Kind kind, IJavaElement javaElement,
			IJavaTypeReference javaTypeReference, Object nonJavaElement,
			String category, String description, String displayName,
			Object feature) {
		this(kind, javaElement, javaTypeReference, nonJavaElement, category,
				description, displayName, feature, false);
	}

	/**
	 * Convenience constructor for setting all encapsulated fields in one place.
	 * 
	 * @param kind
	 *            element kind
	 * @param javaElement
	 *            java element (either the element or the reference are
	 *            required)
	 * @param javaTypeReference
	 *            java type reference (either the reference or the element are
	 *            required)
	 * @param nonJavaElement
	 *            non-Java element
	 * @param category
	 *            category
	 * @param description
	 *            description
	 * @param displayName
	 *            display name
	 * @param feature
	 *            related feature, if any
	 * @param elementOwner
	 *            <tt>true</tt> if the object is the owner of the referenced
	 *            element
	 * @since 0.9
	 */
	protected ItemJavaElementDescriptor(Kind kind, IJavaElement javaElement,
			IJavaTypeReference javaTypeReference, Object nonJavaElement,
			String category, String description, String displayName,
			Object feature, boolean elementOwner) {
		this.kind = kind;
		this.javaElement = javaElement;
		this.javaTypeReference = javaTypeReference;
		this.nonJavaElement = nonJavaElement;
		this.category = category;
		this.description = description;
		this.displayName = displayName;
		this.feature = feature;
		this.elementOwner = elementOwner;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getKind(java.lang.Object)
	 */
	public Kind getKind(Object object) {
		return kind;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getJavaElement(java.lang
	 * .Object)
	 */
	public IJavaElement getJavaElement(Object object) {
		return javaElement;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getJavaTypeReference(
	 * java.lang.Object)
	 */
	public IJavaTypeReference getJavaTypeReference(Object object) {
		return javaTypeReference;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getNonJavaElement(java
	 * .lang.Object)
	 */
	public Object getNonJavaElement(Object object) {
		return nonJavaElement;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getCategory(java.lang
	 * .Object)
	 */
	public String getCategory(Object object) {
		return category;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getDescription(java.lang
	 * .Object)
	 */
	public String getDescription(Object object) {
		return description;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getDisplayName(java.lang
	 * .Object)
	 */
	public String getDisplayName(Object object) {
		return displayName;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getFeature(java.lang.
	 * Object)
	 */
	public Object getFeature(Object object) {
		return feature;
	}

	/**
	 * @see org.eclipse.emf.mint.IItemJavaElementDescriptor2#isElementOwner(java.lang.Object)
	 * @since 0.9
	 */
	public boolean isElementOwner(Object object) {
		return elementOwner;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;

		if (obj == null)
			return false;

		if (getClass() != obj.getClass())
			return false;

		final ItemJavaElementDescriptor other = (ItemJavaElementDescriptor) obj;
		if (kind != other.kind)
			return false;

		if (javaElement == null) {
			if (other.javaElement != null)
				return false;
		} else if (!javaElement.equals(other.javaElement))
			return false;

		if (javaTypeReference == null) {
			if (other.javaTypeReference != null)
				return false;
		} else if (!javaTypeReference.equals(other.javaTypeReference))
			return false;

		if (nonJavaElement == null) {
			if (other.nonJavaElement != null)
				return false;
		} else if (!nonJavaElement.equals(other.nonJavaElement))
			return false;

		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((kind == null ? 0 : kind.hashCode()));
		result = prime * result
				+ ((javaElement == null) ? 0 : javaElement.hashCode());
		result = prime
				* result
				+ ((javaTypeReference == null) ? 0 : javaTypeReference
						.hashCode());
		result = prime * result
				+ ((nonJavaElement == null) ? 0 : nonJavaElement.hashCode());
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		final StringBuilder buf = new StringBuilder(getClass().getSimpleName())
				.append('[');
		buf.append("kind=").append(kind).append(';'); //$NON-NLS-1$
		buf.append("javaElement=").append(javaElement).append(';'); //$NON-NLS-1$
		buf.append("externalTypeReference=").append(javaTypeReference).append( //$NON-NLS-1$
				';');
		buf.append("nonJavaElement=").append(nonJavaElement).append(';'); //$NON-NLS-1$
		buf.append("category=").append(category).append(';'); //$NON-NLS-1$
		buf.append("description=").append(description).append(';'); //$NON-NLS-1$
		buf.append("displayName=").append(displayName).append(';'); //$NON-NLS-1$
		buf.append("feature=").append(feature).append(']'); //$NON-NLS-1$
		return buf.toString();
	}
}
