/*******************************************************************************
 * Copyright (c) 2008 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.internal.ui.search;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.mint.internal.ui.MintUI;
import org.eclipse.search.ui.IQueryListener;
import org.eclipse.search.ui.ISearchQuery;
import org.eclipse.search.ui.ISearchResult;
import org.eclipse.search.ui.ISearchResultListener;
import org.eclipse.search.ui.NewSearchUI;
import org.eclipse.search.ui.SearchResultEvent;
import org.eclipse.search.ui.text.AbstractTextSearchResult;
import org.eclipse.search.ui.text.Match;
import org.eclipse.search.ui.text.MatchEvent;
import org.eclipse.search.ui.text.RemoveAllEvent;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;

public class JavaSearchMonitor implements IQueryListener,
		ISearchResultListener, IResourceChangeListener {

	private static JavaSearchMonitor instance;

	private Map<ISearchResult, Collection<GenModelMatch>> results;

	private Map<IFile, Collection<GenModelMatch>> files;

	private JavaSearchMonitor() {
		NewSearchUI.addQueryListener(this);
		for (ISearchQuery query : NewSearchUI.getQueries())
			queryAdded(query);
	}

	public void queryAdded(ISearchQuery query) {
		if (query.getSearchResult() instanceof AbstractTextSearchResult)
			query.getSearchResult().addListener(this);
	}

	public void queryStarting(ISearchQuery query) {
		// do nothing
	}

	public void queryFinished(ISearchQuery query) {
		// do nothing
	}

	public void queryRemoved(ISearchQuery query) {
		if (query.getSearchResult() instanceof AbstractTextSearchResult) {
			query.getSearchResult().removeListener(this);
			removeMatches(query.getSearchResult());
		}
	}

	public void searchResultChanged(SearchResultEvent event) {
		if (event instanceof RemoveAllEvent) {
			removeMatches(event.getSearchResult());
			return;
		}

		if (event instanceof MatchEvent) {
			ArrayList<GenModelMatch> matches = new ArrayList<GenModelMatch>();
			MatchEvent matchEvent = (MatchEvent) event;
			for (Match match : matchEvent.getMatches()) {
				if (match instanceof GenModelMatch)
					matches.add((GenModelMatch) match);
			}

			ISearchResult result = event.getSearchResult();
			if (matchEvent.getKind() == MatchEvent.ADDED) {
				addMatches(result, matches);
			} else {
				removeMatches(result, matches);
			}

			return;
		}
	}

	public void resourceChanged(IResourceChangeEvent event) {
		synchronized (this) {
			if (files == null)
				return;
		}

		final ArrayList<GenModelMatch> matches = new ArrayList<GenModelMatch>();
		IResourceDelta delta = event.getDelta();
		try {
			delta.accept(new IResourceDeltaVisitor() {
				public boolean visit(IResourceDelta delta) throws CoreException {
					synchronized (JavaSearchMonitor.this) {
						if (files == null)
							return false;
					}

					IResource resource = delta.getResource();
					int kind = delta.getKind();
					if (resource.getType() == IResource.FILE) {
						if (kind == IResourceDelta.REMOVED)
							collectRemoved((IFile) resource, matches);

						return false;
					}

					if (kind == IResourceDelta.ADDED)
						return false;

					return true;
				}
			});
		} catch (CoreException e) {
			MintUI.getDefault().logError(null, e);
		}

		notifyRemoved(matches);
	}

	private synchronized void collectRemoved(IFile file,
			Collection<GenModelMatch> matches) {
		Collection<GenModelMatch> removed = files.remove(file);
		if (removed != null) {
			matches.addAll(removed);
			if (files.isEmpty()) {
				ResourcesPlugin.getWorkspace().removeResourceChangeListener(
						JavaSearchMonitor.this);
				files = null;
			}
		}
	}

	private void notifyRemoved(Collection<GenModelMatch> matches) {
		if (matches.isEmpty())
			return;

		Match[] matchArray = matches.toArray(new Match[matches.size()]);
		ISearchQuery[] queries = NewSearchUI.getQueries();
		for (ISearchQuery query : queries) {
			if (!(query.getSearchResult() instanceof AbstractTextSearchResult))
				continue;

			((AbstractTextSearchResult) query.getSearchResult())
					.removeMatches(matchArray);
		}

		HashSet<Resource> resources = new HashSet<Resource>();
		for (GenModelMatch match : matches)
			resources.add(match.getResource());

		unload(resources);
	}

	private synchronized void addMatches(ISearchResult result,
			Collection<GenModelMatch> matches) {
		if (results == null)
			results = new HashMap<ISearchResult, Collection<GenModelMatch>>();

		Collection<GenModelMatch> allMatches = results.get(result);
		if (allMatches == null) {
			allMatches = new HashSet<GenModelMatch>(2);
			results.put(result, allMatches);
		}

		allMatches.addAll(matches);

		if (files == null) {
			files = new HashMap<IFile, Collection<GenModelMatch>>();
			ResourcesPlugin.getWorkspace().addResourceChangeListener(this,
					IResourceChangeEvent.POST_CHANGE);
		}

		for (GenModelMatch match : matches) {
			allMatches = files.get(match.getFile());
			if (allMatches == null) {
				allMatches = new HashSet<GenModelMatch>(2);
				files.put(match.getFile(), allMatches);
			}

			allMatches.add(match);
		}
	}

	private synchronized void removeMatches(ISearchResult result,
			Collection<GenModelMatch> matches) {
		if (results != null) {
			Collection<GenModelMatch> allMatches = results.get(result);
			if (allMatches != null) {
				if (allMatches.removeAll(matches) && allMatches.isEmpty()) {
					results.remove(result);
					if (results.isEmpty())
						results = null;
				}
			}
		}

		removeMatchFiles(matches);
	}

	private void removeMatchFiles(Collection<GenModelMatch> matches) {
		HashSet<Resource> resources = new HashSet<Resource>();
		synchronized (this) {
			if (files != null) {
				for (GenModelMatch match : matches) {
					IFile file = match.getFile();
					Collection<GenModelMatch> allMatches = files.get(file);
					if (allMatches != null) {
						if (allMatches.remove(match) && allMatches.isEmpty()) {
							files.remove(file);
							resources.add(match.getResource());
							if (files.isEmpty()) {
								ResourcesPlugin.getWorkspace()
										.removeResourceChangeListener(this);
								files = null;
								break;
							}
						}
					}
				}

			}
		}

		if (!resources.isEmpty())
			unload(resources);
	}

	private synchronized void removeMatches(ISearchResult result) {
		if (results != null) {
			Collection<GenModelMatch> allMatches = results.remove(result);
			if (allMatches != null) {
				removeMatchFiles(allMatches);
				if (results.isEmpty())
					results = null;
			}
		}
	}

	private void dispose() {
		NewSearchUI.removeQueryListener(this);
		synchronized (this) {
			if (files != null) {
				ResourcesPlugin.getWorkspace().removeResourceChangeListener(
						this);
				files = null;
			}

			if (results != null) {
				for (ISearchResult result : results.keySet())
					result.removeListener(this);

				results = null;
			}
		}
	}

	private void unload(final Collection<Resource> resources) {
		Display display = PlatformUI.getWorkbench().getDisplay();
		if (display != null && !display.isDisposed())
			display.asyncExec(new Runnable() {
				public void run() {
					for (Resource resource : resources)
						resource.unload();
				}
			});
	}

	public static synchronized void ensureStarted() {
		if (instance == null)
			instance = new JavaSearchMonitor();
	}

	public static synchronized void stop() {
		if (instance != null) {
			instance.dispose();
			instance = null;
		}
	}
}
