/*******************************************************************************
 * Copyright (c) 2009 - 2010 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.internal.ui.actions;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Properties;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.edit.domain.AdapterFactoryEditingDomain;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.mint.IItemJavaElementDescriptor;
import org.eclipse.emf.mint.IItemJavaElementDescriptor2;
import org.eclipse.emf.mint.IItemJavaElementSource;
import org.eclipse.emf.mint.IJavaTypeReference;
import org.eclipse.emf.mint.internal.ui.MintUI;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.refactoring.IJavaRefactorings;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.ChangeDescriptor;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.ltk.core.refactoring.Refactoring;
import org.eclipse.ltk.core.refactoring.RefactoringChangeDescriptor;
import org.eclipse.ltk.core.refactoring.RefactoringContribution;
import org.eclipse.ltk.core.refactoring.RefactoringCore;
import org.eclipse.ltk.core.refactoring.RefactoringDescriptor;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;

public class CleanGeneratedRefactoring extends Refactoring {

	private final Object[] elements;

	private Refactoring delegate;

	public CleanGeneratedRefactoring(Object[] elements) {
		this.elements = elements;
	}

	@Override
	public String getName() {
		return "Clean Generated Artifacts";
	}

	@Override
	public RefactoringStatus checkInitialConditions(IProgressMonitor monitor)
			throws CoreException, OperationCanceledException {
		RefactoringStatus status = new RefactoringStatus();
		if (delegate == null)
			delegate = createDelegate(status, monitor);

		status.merge(delegate.checkInitialConditions(monitor));
		return status;
	}

	private Refactoring createDelegate(RefactoringStatus status,
			IProgressMonitor monitor) throws CoreException {
		Properties args = new Properties();
		LinkedHashSet<IJavaElement> elements = new LinkedHashSet<IJavaElement>();
		LinkedHashSet<IResource> resources = new LinkedHashSet<IResource>();

		ComposedAdapterFactory adapterFactory = new ComposedAdapterFactory(
				ComposedAdapterFactory.Descriptor.Registry.INSTANCE);

		UniqueEList.FastCompare<EObject> objects = new UniqueEList.FastCompare<EObject>();
		for (Object element : this.elements) {
			Object root = AdapterFactoryEditingDomain.unwrap(element);
			if (root instanceof EObject) {
				objects.add((EObject) root);
				for (Iterator<EObject> i = ((EObject) root).eAllContents(); i
						.hasNext();)
					objects.add(i.next());
			}
		}

		monitor.beginTask("Collecting generated artifacts", objects.size());
		try {
			for (Object element : objects) {
				Object adapter = adapterFactory.adapt(element,
						IItemJavaElementSource.class);
				if (adapter instanceof IItemJavaElementSource) {
					IItemJavaElementSource src = (IItemJavaElementSource) adapter;
					List<IItemJavaElementDescriptor> descs = src
							.getJavaElementDescriptors(element);
					for (IItemJavaElementDescriptor desc : descs) {
						if (!(desc instanceof IItemJavaElementDescriptor2)
								|| !((IItemJavaElementDescriptor2) desc)
										.isElementOwner(element))
							continue;

						IJavaElement je = null;
						IResource resource = null;
						switch (desc.getKind(element)) {
						case JAVA_ELEMENT:
							je = desc.getJavaElement(element);
							monitor.worked(1);
							break;
						case JAVA_TYPE_REFERENCE:
							IJavaTypeReference ref = desc
									.getJavaTypeReference(element);
							try {
								je = ref.getContext().findType(
										ref.getTypeName(),
										new SubProgressMonitor(monitor, 1));
							} catch (JavaModelException e) {
								// ignore inaccessible elements
								continue;
							}

							break;
						case NON_JAVA_RESOURCE:
							Object obj = desc.getNonJavaElement(element);
							if (obj instanceof IResource)
								resource = (IResource) obj;
							else if (obj instanceof IAdaptable)
								resource = (IResource) ((IAdaptable) obj)
										.getAdapter(IResource.class);

							monitor.worked(1);
							break;
						default:
							monitor.worked(1);
						}

						if (je != null && je.exists() && !je.isReadOnly()
								&& !isIncluded(elements, je))
							elements.add(je);

						if (resource != null && resource.exists())
							resources.add(resource);
					}
				} else {
					monitor.worked(1);
				}
			}
		} finally {
			monitor.done();
			adapterFactory.dispose();
		}

		int i = 0;
		for (IResource resource : resources) {
			args.setProperty("element" + (++i), resource.getFullPath()
					.toPortableString());
			args.setProperty("name" + i, resource.getName());
		}

		for (IJavaElement je : elements) {
			args.setProperty("element" + (++i), je.getHandleIdentifier());
			args.setProperty("name" + i, je.getElementName());
		}

		args.setProperty("elements", String.valueOf(elements.size()));
		args.setProperty("resources", String.valueOf(resources.size()));
		args.setProperty("accessors", Boolean.toString(false));
		args.setProperty("subPackages", Boolean.toString(false));

		RefactoringContribution contrib = RefactoringCore
				.getRefactoringContribution(IJavaRefactorings.DELETE);
		if (contrib == null)
			throw new CoreException(new Status(IStatus.ERROR, MintUI.PLUGIN_ID,
					"Unable to create refactoring delegate."));

		RefactoringDescriptor desc = contrib.createDescriptor(
				IJavaRefactorings.DELETE, null, "Delete Generated Artifacts",
				null, args, RefactoringDescriptor.STRUCTURAL_CHANGE
						| RefactoringDescriptor.MULTI_CHANGE);
		return desc.createRefactoring(status);
	}

	private boolean isIncluded(Collection<IJavaElement> elements,
			final IJavaElement je) {
		for (IJavaElement element : elements) {
			IJavaElement parent = je;
			do {
				if (parent.equals(element))
					return true;
			} while ((parent = parent.getParent()) != null);
		}

		return false;
	}

	@Override
	public RefactoringStatus checkFinalConditions(IProgressMonitor monitor)
			throws CoreException, OperationCanceledException {
		return delegate.checkFinalConditions(monitor);
	}

	@Override
	public Change createChange(IProgressMonitor monitor) throws CoreException,
			OperationCanceledException {
		CleanGeneratedChange change = new CleanGeneratedChange();
		change.add(delegate.createChange(monitor));
		return change;
	}

	private class CleanGeneratedChange extends CompositeChange {

		private ChangeDescriptor descriptor;

		public CleanGeneratedChange() {
			super("Clean Generated Artifacts");
			markAsSynthetic();
		}

		@Override
		public ChangeDescriptor getDescriptor() {
			if (descriptor == null)
				descriptor = new RefactoringChangeDescriptor(
						new CleanGeneratedRefactoringDescriptor(elements));

			return descriptor;
		}
	}
}
