/*******************************************************************************
 * Copyright (c) 2008, 2009 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.util.EcoreUtil;

/**
 * Adapter for tracking changes to a group of related objects. Each time a
 * related object is added to the group (typically through a reference), changes
 * to its features are tracked and notifications forwarded to registered
 * listeners. This is particularly useful when an object's derived feature
 * depends on a feature that belongs to one of its referenced objects.
 * 
 * <p>
 * Clients may extend this class.
 * </p>
 */
public abstract class GroupAdapterImpl extends AdapterImpl {

	/**
	 * Adapter used to forward notifications from related objects to the root
	 * object.
	 * 
	 * @see GroupAdapterImpl#createNotificationForwarder()
	 */
	protected class NotificationForwarder extends AdapterImpl {

		/* (non-Javadoc)
		 * @see org.eclipse.emf.common.notify.impl.AdapterImpl#setTarget(org.eclipse.emf.common.notify.Notifier)
		 */
		@Override
		public void setTarget(Notifier target) {
			super.setTarget(target);
			if (target instanceof EObject)
				addGroupTargets(getGroupTargets((EObject) target));
		}

		/* (non-Javadoc)
		 * @see org.eclipse.emf.common.notify.impl.AdapterImpl#unsetTarget(org.eclipse.emf.common.notify.Notifier)
		 */
		@Override
		public void unsetTarget(Notifier target) {
			if (target instanceof EObject)
				removeGroupTargets(getGroupTargets((EObject) target));

			super.unsetTarget(target);
		}

		/* (non-Javadoc)
		 * @see org.eclipse.emf.common.notify.impl.AdapterImpl#isAdapterForType(java.lang.Object)
		 */
		@Override
		public boolean isAdapterForType(Object type) {
			return type == GroupAdapterImpl.this;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.emf.common.notify.impl.AdapterImpl#notifyChanged(org.eclipse.emf.common.notify.Notification)
		 */
		@Override
		public void notifyChanged(Notification msg) {
			GroupAdapterImpl.this.notifyChanged(msg);
		}

		/**
		 * Disposes this adapter by removing it from its target.
		 */
		public void dispose() {
			Notifier oldTarget = target;
			target = null;
			if (oldTarget != null)
				oldTarget.eAdapters().remove(this);
		}
	}

	/**
	 * Simple list for tracking notifiers.
	 * 
	 * @deprecated no longer used
	 */
	@SuppressWarnings("serial")
	protected static class NotifierEList extends
			BasicEList.FastCompare<Notifier> {
		
		/* (non-Javadoc)
		 * @see org.eclipse.emf.common.util.AbstractEList#canContainNull()
		 */
		@Override
		protected boolean canContainNull() {
			return false;
		}
	}

	/**
	 * Tracked targets (the group).
	 * 
	 * @deprecated this adapter only tracks a single target
	 */
	protected NotifierEList targets;

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.common.notify.impl.AdapterImpl#setTarget(org.eclipse.
	 * emf.common.notify.Notifier)
	 */
	@Override
	public void setTarget(Notifier target) {
		super.setTarget(target);
		if (target instanceof EObject)
			addGroupTargets(getGroupTargets((EObject) target));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.common.notify.impl.AdapterImpl#unsetTarget(org.eclipse
	 * .emf.common.notify.Notifier)
	 */
	@Override
	public void unsetTarget(Notifier target) {
		if (target instanceof EObject)
			removeGroupTargets(getGroupTargets((EObject) target));

		super.unsetTarget(target);
	}

	/**
	 * Disposes the adapter by removing it from its target.
	 */
	public void dispose() {
		Notifier oldTarget = target;
		target = null;
		if (oldTarget != null)
			oldTarget.eAdapters().remove(this);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.common.notify.impl.AdapterImpl#notifyChanged(org.eclipse
	 * .emf.common.notify.Notification)
	 */
	public void notifyChanged(Notification msg) {
		Object notifier = msg.getNotifier();
		Object feature = msg.getFeature();
		if (notifier instanceof EObject && feature instanceof EReference) {
			EObject object = (EObject) notifier;
			EReference ref = (EReference) feature;
			switch (msg.getEventType()) {
			case Notification.ADD:
				if (isGroupReference(object, ref)) {
					List<EObject> values = Collections
							.singletonList((EObject) msg.getNewValue());
					addGroupTargets(values);
				}

				break;
			case Notification.ADD_MANY:
				if (isGroupReference(object, ref)) {
					@SuppressWarnings("unchecked")
					List<EObject> values = (List<EObject>) msg.getNewValue();
					addGroupTargets(values);
				}

				break;
			case Notification.REMOVE:
				if (isGroupReference(object, ref)) {
					List<EObject> values = Collections
							.singletonList((EObject) msg.getOldValue());
					removeGroupTargets(values);
				}

				break;
			case Notification.REMOVE_MANY:
				if (isGroupReference(object, ref)) {
					@SuppressWarnings("unchecked")
					List<EObject> values = (List<EObject>) msg.getOldValue();
					removeGroupTargets(values);
				}

				break;
			case Notification.SET:
				if (isGroupReference(object, ref)) {
					if (msg.getOldValue() != null) {
						List<EObject> values = Collections
								.singletonList((EObject) msg.getOldValue());
						removeGroupTargets(values);
					}

					if (msg.getNewValue() != null) {
						List<EObject> values = Collections
								.singletonList((EObject) msg.getNewValue());
						addGroupTargets(values);
					}
				}

				break;
			case Notification.UNSET:
				if (isGroupReference(object, ref) && msg.getOldValue() != null) {
					List<EObject> values = Collections
							.singletonList((EObject) msg.getOldValue());
					removeGroupTargets(values);
				}

				break;
			}
		}
	}

	/**
	 * Creates a new @{link NotificationForwarder}.
	 * 
	 * @return new notification forwarder
	 */
	protected NotificationForwarder createNotificationForwarder() {
		return new NotificationForwarder();
	}

	/**
	 * Adds the given targets to the group of tracked targets.
	 * 
	 * @param groupTargets
	 *            new targets to track
	 */
	protected void addGroupTargets(List<? extends Notifier> groupTargets) {
		for (Notifier groupTarget : groupTargets) {
			if (groupTarget == target)
				continue;

			if (EcoreUtil.getExistingAdapter(groupTarget, this) == null)
				groupTarget.eAdapters().add(createNotificationForwarder());
		}
	}

	/**
	 * Removes the targets from the group of tracked targets.
	 * 
	 * @param groupTargets
	 *            targets to remove from the group
	 */
	protected void removeGroupTargets(List<? extends Notifier> groupTargets) {
		for (Notifier groupTarget : groupTargets) {
			ArrayList<Adapter> adapters = new ArrayList<Adapter>();
			for (Adapter adapter : groupTarget.eAdapters())
				if (adapter.isAdapterForType(this))
					adapters.add(adapter);
			
			groupTarget.eAdapters().removeAll(adapters);
		}
	}

	/**
	 * Returns the object's direct relations.
	 * 
	 * @param object
	 *            object whose direct relations to return
	 * @return object's direct relations
	 */
	protected List<EObject> getGroupTargets(EObject object) {
		BasicEList.FastCompare<EObject> list = new BasicEList.FastCompare<EObject>();
		List<EReference> refs = getGroupReferences(object);
		for (EReference ref : refs) {
			Object value = object.eGet(ref);
			if (ref.isMany()) {
				@SuppressWarnings("unchecked")
				EList<EObject> values = (EList<EObject>) value;
				list.addAll(values);
			} else if (value != null) {
				list.add((EObject) value);
			}
		}

		return list;
	}

	/**
	 * Determines whether the object's reference participates in defining its
	 * relations.
	 * 
	 * @param object
	 *            object
	 * @param ref
	 *            object's reference
	 * @return {@code true} if the given reference contributes its direct
	 *         relations
	 */
	protected boolean isGroupReference(EObject object, EReference ref) {
		List<EReference> list = getGroupReferences(object);
		if (list != null)
			return list.contains(ref);

		return false;
	}

	/**
	 * Returns the object's references that define its direct relations.
	 * 
	 * @param object
	 *            object whose references to return
	 * @return object's references that define its direct relations
	 */
	protected abstract List<EReference> getGroupReferences(EObject object);

	/**
	 * Returns the group's root object.
	 * 
	 * @return the group's root object
	 * 
	 * @deprecated use {@link #getTarget()} instead
	 */
	protected Notifier getRootTarget() {
		return getTarget();
	}
}
