/*******************************************************************************
 * Copyright (c) 2005 - 2006 Joel Cheuoua & others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Joel Cheuoua - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.codegen.jet.editor;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import org.osgi.framework.Bundle;

/**
 * @author Joel Cheuoua
 */
public class JETURLClassLoader extends URLClassLoader {
  private List delegateLoaders = new ArrayList();
  private List delegateBundles = new ArrayList();
  
  public JETURLClassLoader(File[] files) {
    this(new URL[0]);
    setFiles(files);
  }

  public JETURLClassLoader(File[] files, ClassLoader parent) {
    this(new URL[0], parent);
    setFiles(files);
  }

  public JETURLClassLoader(URL[] urls) {
    super(urls);
  }

  public JETURLClassLoader(URL[] urls, ClassLoader parent) {
    super(urls, parent);
  }

  public void addClassLoaderDelegate(ClassLoader delegate) {
    delegateLoaders.add(delegate);
  }

  public void addBundleDelegate(Bundle delegate) {
    delegateBundles.add(delegate);
  }

  /**
   * Sets for the specified files to this class loader
   * 
   * @param files
   *          The files from which to load classes and resources
   */
  public synchronized void setFiles(File[] files) {
    if (files == null)
      return;
    for (int i = 0; i < files.length; i++) {
      File file = files[i];
      String[] filenames;
      filenames = SimpleFileFilter.fileOrFiles(file);
      if (filenames == null)
        continue;
      for (int j = 0; j < filenames.length; j++) {
        file = new File(filenames[j]);
        if (file.canRead() && (file.isDirectory() || isZipOrJarArchive(file))) {
          // Transforms the file into an url
          try {
            String path = file.getAbsolutePath();
            file = new File(path);
            addURL(file.toURL());
          } catch (MalformedURLException e) {
          }
        }
      }
    }
  }

  /**
   * Returns an array of files that has been set to this class loader
   */
  public File[] getFiles() {
    URL[] urls = getURLs();
    File[] result = new File[urls.length];
    for (int i = 0; i < urls.length; i++) {
      result[i] = new File(urls[i].getFile());
    }
    return result;
  }

  /**
   * Tests if a file is a ZIP or JAR archive.
   * 
   * @param file
   *          the file to be tested.
   * @return <code>true</code> if the file is a ZIP/JAR archive, <code>false</code> otherwise.
   */
  private boolean isZipOrJarArchive(File file) {
    boolean isArchive = true;
    ZipFile zipFile = null;
    try {
      zipFile = new ZipFile(file);
    } catch (ZipException zipCurrupted) {
      isArchive = false;
    } catch (IOException anyIOError) {
      isArchive = false;
    } finally {
      if (zipFile != null) {
        try {
          zipFile.close();
        } catch (IOException e) {
        }
      }
    }
    return isArchive;
  }

  /*
   * (non-Javadoc)
   * 
   * @see java.lang.ClassLoader#loadClass(java.lang.String)
   */
  public Class loadClass(String name) throws ClassNotFoundException {
    Class c = null;
    // look in the delegates if any
    if (delegateLoaders != null && !delegateLoaders.isEmpty()) {
      for (Iterator iter = delegateLoaders.iterator(); iter.hasNext();) {
        ClassLoader delegate = (ClassLoader) iter.next();
        try {
          c = delegate.loadClass(name);
          return c;
        } catch (ClassNotFoundException subex) {
          // Continue to the next delegate
        }
      }
    }   
    if (delegateBundles != null && !delegateBundles.isEmpty()) {
      for (Iterator iter = delegateBundles.iterator(); iter.hasNext();) {
        Bundle delegate = (Bundle) iter.next();
        try {
          c = delegate.loadClass(name);
          return c;
        } catch (ClassNotFoundException subex) {
          // Continue to the next delegate
        }
      }
    }
    return super.loadClass(name);
  }

  public static class SimpleFileFilter implements FilenameFilter {
    private String[] extensions;

    public SimpleFileFilter(String ext) {
      this(new String[] { ext });
    }

    public SimpleFileFilter(String[] exts) {
      extensions = new String[exts.length];
      for (int i = 0; i < exts.length; i++) {
        extensions[i] = exts[i].toLowerCase();
      }
    }

    /** filenamefilter interface method */
    public boolean accept(File dir, String _name) {
      String name = _name.toLowerCase();
      for (int i = 0; i < extensions.length; i++) {
        if (name.endsWith(extensions[i]))
          return true;
      }
      return false;
    }

    /**
     * this method checks to see if an asterisk is embedded in the filename, if it is, it does an "ls" or "dir" of the
     * parent directory returning a list of files that match eg. /usr/home/mjennings/*.jar would expand out to all of
     * the files with a .jar extension in the /usr/home/mjennings directory
     */
    public static String[] fileOrFiles(File f) {
      if (f == null)
        return null;
      String fname = f.getName();
      String[] files;
      if (fname.length() > 0 && fname.charAt(0) == '*') {
        String parentName = f.getParent();
        if (parentName == null)
          return null;
        File parent = new File(parentName);
        String filter = fname.substring(1, fname.length());
        files = parent.list(new SimpleFileFilter(filter));
        return files;
      } else {
        files = new String[1];
        files[0] = f.getPath();// was:fname;
        return files;
      }
    }
  }
}
